module NEB_toolbox

  use math_head
  use MatrixOps

  implicit none

  contains

    subroutine set_path(path)
      real(DOUBLE),dimension(0:nimages+1,fulldim) :: path
      integer :: i

      do i=nroot-nimages+1,nroot
         path(i-(nroot-nimages+1)+1,1:fulldim) =  root(i)%p%pos(1:fulldim)
      end do

    end subroutine set_path

   subroutine alignment(path,out)
     integer :: i
     real(DOUBLE),dimension(0:nimages+1,ndim) :: path
     real(DOUBLE),dimension(ndim) :: a,b
     real(DOUBLE),dimension(nimages) :: out

     do i=1,nimages
        a = path(i,:)-path(i-1,:);        b = path(i+1,:)-path(i,:)
        call normalize(a,ndim);       call normalize(b,ndim)
        out(i) = dot_product(a,b)
     end do
     
   end subroutine alignment

   ! Code translated from:  http://math.fullerton.edu/mathews/n2003/Web/CubicSplinesMod/CubicSplinesMod.html
   subroutine spaceoutcubic(path,n)
     integer :: nogood,n, i,ind, j, nimages,lastgood ! n = nimages + 2
     real(DOUBLE),allocatable,dimension(:) :: dists, lpath
     real(DOUBLE),dimension(n,ndim) :: path
     real(DOUBLE),allocatable,dimension(:,:,:) :: coeffs
     real(DOUBLE),dimension(n-2) :: aligned
     integer,dimension(n-2) :: keep
     real(DOUBLE) :: space,pathlen,dx(ndim)

     nimages = n-2

     call alignment(path,aligned)
     nogood = 0;
!     print*, "alignment in spaceoutcubic ", aligned

     ! *** Correct for badly aligned points. ***
     do i=1,nimages
        if(aligned(i) > 0.6) then
           keep(i) = 1
        else 
           print*, "Skipping point ", i
           keep(i) = 0;   nogood = nogood + 1
        end if
     end do

     ! Get the distances between good points.
     allocate(dists(n-nogood))
     allocate(lpath(n-nogood))
     allocate(coeffs(ndim,n-1-nogood,4))
     dists(1) = 0

     ind = 1;     pathlen=0; lastgood = 1
     do i=2,nimages+1
        if (keep(i-1)==1) then
           pathlen = pathlen + norm(path(i,:) - path(lastgood,:),ndim)
           dists(ind+1) = pathlen
           ind = ind + 1
           lastgood = i
        end if
     end do

     dists(ind+1) = pathlen + norm(path(nimages+2,:)-path(lastgood,:),ndim)

     space = dists(ind+1)/(nimages+1);

     do i=1,ndim
        ind=1
        do j=1,nimages+2
           if (j==1.or.j==nimages+2) then
              lpath(ind) = path(j,i);
              ind = ind + 1
           else 
              if (keep(j-1)==1) then
                 lpath(ind) = path(j,i);
                 ind = ind + 1
              end if
           end if
        end do
        call get_cubic_coeffs(dists,lpath,n-nogood,coeffs(i,:,:))
     end do

     do i=1,nimages
        do j=1,ndim
           call get_cubic_values(dists,space*i,n-nogood,coeffs(j,:,:),path(i+1,j))
        end do
     end do

     deallocate(dists,lpath,coeffs)

   end subroutine spaceoutcubic

   subroutine fullNCS(coeffs,x,t,n,out)
     integer :: i,j,n
     real(DOUBLE) :: t
     real(DOUBLE),dimension(n) :: x
     real(DOUBLE),dimension(ndim,n,4) :: coeffs
     real(DOUBLE),dimension(ndim) :: out

     i = 1

     do while ( t > x(i) .and. i < n )
        i = i + 1
     end do
     i = i - 1

     do j=1,ndim
        out(j) = coeffs(j,i,1)+(t-x(i))*coeffs(j,i,2) + coeffs(j,i,3)*(t - x(i))**2 + &
             coeffs(j,i,4)*(t-x(i))**3
     end do
   end subroutine fullNCS

   subroutine get_cubic_coeffs(x,y,np1,s)
     integer :: n,np1,k
     real(DOUBLE) :: t
     real(DOUBLE), dimension(np1) :: x,y,m
     real(DOUBLE), dimension(np1-1,4) :: s
     real(DOUBLE), dimension(np1-1) :: h,d
     real(DOUBLE), dimension(np1-2) :: a,b,c,v

     n = np1 - 1;

     m=0;
     s=0
     a=0;b=0;c=0;v=0
     h=0;d=0

     h(1) = x(2) - x(1)
     d(1) = (y(2) - y(1))/h(1)

     do k=2,n
        h(k) = x(k+1) - x(k)
        d(k) = (y(k+1) - y(k))/h(k)
        a(k-1) = h(k)
        b(k-1) = 2d0*(h(k-1) + h(k))
        c(k-1) = h(k)
        v(k-1) = 6d0*(d(k) - d(k-1))
     end do

     do k=2,n-1
        t= a(k-1)/b(k-1)
        b(k) = b(k) - t * c(k-1)
        v(k) = v(k) - t * v(k-1)
     end do
     m(n) = v(n-1)/b(n-1)

     do k=n-2,1,-1
        m(k+1) = (v(k) - c(k)*m(k+2)) / b(k)
     end do

     do k=1,n
        s(k,1) = y(k)
        s(k,2) = d(k) - 1d0/6d0 * h(k) * (2d0*m(k) + m(k+1))
        s(k,3) = m(k)/2d0
        s(k,4) = (m(k+1)-m(k))/(6d0*h(k))
     end do

   end subroutine get_cubic_coeffs

   subroutine get_cubic_values(x,t,np1,s,out)
     integer :: j,n,k,np1
     real(DOUBLE) :: w,out,t
     real(DOUBLE), dimension(np1) :: x
     real(DOUBLE), dimension(np1-1,4) :: s

     k = -1;
     n = np1 - 1;

     do j=1,n
        if(x(j) <= t .and. t < x(j+1)) then
           k = j
        end if
     end do

     if(t<x(1)) then
        k=1
     else if (t>=x(n+1)) then
        k=n
     end if

     w = t - x(k)
     out = ( (s(k,4)*w + s(k,3))*w + s(k,2) )*w + s(k,1)
     
   end subroutine get_cubic_values


   subroutine funupwind(path,dE,energy,nimages,ndim,gtan)
     integer :: nimages,i,ndim
     real(DOUBLE), dimension(0:nimages+1,ndim) :: path
     real(DOUBLE), dimension(nimages,ndim) :: dE,gtan, tangents
     real(DOUBLE), dimension(nimages) :: energy

     call calc_tangents(path,nimages,ndim,tangents,energy,0)

     do i=1,nimages
        gtan(i,:) = dE(i,:) - sum(dE(i,:)*tangents(i,:))*tangents(i,:)
     end do

   end subroutine funupwind

   subroutine calc_tangents(path,nimages,ndim,tangents,energy,tan_choice)
     integer :: nimages,i,tan_choice,ndim
     real(DOUBLE), dimension(0:nimages+1,ndim) :: path
     real(DOUBLE), dimension(nimages,ndim) :: tangents
     real(DOUBLE), dimension(nimages) :: energy
     real(DOUBLE), dimension(0:nimages+1) :: V
     real(DOUBLE), dimension(ndim) :: tp,tm
     real(DOUBLE) :: Vmax,Vmin

     if (tan_choice == 1) then
        do i = 1, nimages
           tangents(i,:) = path(i+1,:) - path(i-1,:)
           tangents(i,:) = tangents(i,:)/sqrt(sum(tangents(i,:)**2))
        end do
     else
        V(0) =-1d10
        V(1:nimages) = energy
        V(nimages+1) =-1d10

        do i = 1, nimages
           tp = path(i+1,:) - path(i,:)
           tm = path(i,:) - path(i-1,:)
           Vmax = max( abs(V(i+1)-V(i)), abs(V(i-1)-V(i)))
           Vmin = min( abs(V(i+1)-V(i)), abs(V(i-1)-V(i)))
           
           if (V(i+1) >= V(i) .and. V(i) >= V(i-1)) then
              tangents(i,:) = tp
           else if (V(i+1) <= V(i) .and. V(i) <= V(i-1)) then
              tangents(i,:) = tm
           else if (V(i+1) > V(i-1)) then
              tangents(i,:) = tp*Vmax+tm*Vmin
           else
              tangents(i,:) = tp*Vmin+tm*Vmax
           endif
           
           tangents(i,:) = tangents(i,:)/sqrt(sum(tangents(i,:)**2))
        end do
     end if
   end subroutine calc_tangents

   ! **** Create temp directories for each point on the path ****
   subroutine make_dirs(nimages) 
     integer :: i,nimages
     character(80) :: cmd

     do i=1,nimages
        write(cmd,'(A16,I3.3)') 'mkdir -p point',i
        call system(cmd)
     end do

   end subroutine make_dirs


   subroutine print_path(path,images)
     integer :: i,images
     real(DOUBLE),dimension(images,fulldim) :: path
     
     do i=1,images
        print*, "image: ",i
        print*, path(i,:)
     end do
   end subroutine print_path

   subroutine eqconst(path,out)
     use math_head
     integer :: i
     real(DOUBLE),dimension(0:nimages+1,ndim) :: path
     real(DOUBLE),dimension(nimages+1) :: dists
     real(DOUBLE) :: avg_dist,out

     do i=1,nimages+1
        dists(i) = sqrt(sum((path(i,:)-path(i-1,:))**2))
     end do

     avg_dist = sum(dists)/(nimages+1)

     out = maxval(abs(dists-avg_dist))

   end subroutine eqconst


end module NEB_toolbox
