import re
import sys,os
import math
from copy import *
from numpy import *
from math_fun import *

class file_fun:
    def __init__(self,filename,ff2):
        self.filename = filename
        self.gfilename = ff2

    def write_single_resp_fit(self,gl,name):
        charge = gl.get_charge()
        elements = gl.get_elements()
        natoms = len(elements)

        # Read in the index and charges of the ACE and NME caps
        num = 6
        try:
            f = open(self.filename,'r')
            line = f.readline();        L = line.split()
            ace_index = map(int,L[0:num])
            line = f.readline();        L = line.split()
            ace_charge = map(float,L[0:num])
            line = f.readline();        L = line.split()
            nme_index = map(int,L[0:num])
            line = f.readline();        L = line.split()
            nme_charge = map(float,L[0:num])
            f.close()
        except IOError:
            print "index.dat file not found. Assuming no fixed charges."
            ace_index = []; ace_charge = []
            nme_index = []; nme_charge = []

        # Read in charge groups.
        gcharges = []; gindex = []
        try:
            f = open(self.gfilename,'r')
            for line in f:
                L = line.split()
                gcharges.append(L[0])
                gindex.append(L[1:])
        except IOError:
            print self.gfilename + " not found. Assuming no charge groups."

        # Merge the lists into one.
        index = ace_index
        for line in nme_index: index.append(line)
        charges = ace_charge
        for line in nme_charge: charges.append(line)

        # Print out the RESP input file.
        f = open(name + ".in",'w')
        f.write("ln5 RESP run #1\n &cntrl\n ihfree=1,\n qwt=0.0005,\n iqopt=2\n /\n")
        f.write("    1.0\nlink\n")
        f.write('{0:5d}{1:5d}\n'.format(charge, natoms))
        for i in range(natoms):
            ind = -1 if (i in index) else 0
            f.write('{0:5d}{1:5d}\n'.format(gl.atoms[i].elnum+1, ind))
        for k in range(len(gcharges)):
            nn = int((len(gindex[k])-1) / 8)
            f.write("{0:5d}  {1: 5.5f}\n".format(len(gindex[k]),float(gcharges[k])))
            for i in range(nn+1):
                for j in range(max(len(gindex[k])-i*8,8)):
                    f.write("{0:5d}{1:5d}".format(1,int(gindex[k][j+i*8])+1))
                f.write('\n')
        f.write('\n')
        f.close()
        
        # Write out the qin file.
        f = open(name + ".qin",'w')
        for i in range(natoms):
            ch = 0.0
            for j in range(len(index)):
                if (index[j]==i): ch = charges[j]
            f.write(' {0: .6f}'.format(ch))
            if ((i+1) % 8==0): f.write('\n')
#        if (not (natoms % 8==0)): f.write('\n')
#        f.write('\n')
        f.close()

class LOT_dir:
    def __init__(self,dirs,nscan,nref):
        self.dirs = dirs
        self.nscan = nscan
        self.nref = nref

    def read_ref_energy(self,dir,name):
        # Read in the reference energies.
        energy = []
        for i in range(self.nscan):
            log_file = "./{0}/{1}{2}.log".format(dir,name,i)
            print "log_file: ", log_file
            gau = gaussian_log_file(log_file)
            vals, sinfo, en, xyz = gau.read_scan_info()
            energy.append([])
            for j in range(len(en)):
                energy[i].append(627.51*(en[j]-en[0]))
#                print "energy: ",j, energy[i][j]
        return energy

    def read_ref_energies(self):
        energy = {}
        for dir in self.dirs[0:self.nref-1]:
            energy[dir] = {}
            for fix in ["","BONDS","ANGLES"]:
                energy[dir][fix] = self.read_ref_energy(dir,"dihed_ref" + fix)

        return energy

    def read_all_profiles(self):
        energy = {}
        for dir in self.dirs[0:self.nref]:
            energy[dir] = self.read_ref_energy(dir,"dihed")

        return energy

    def print_profiles(self,energy):
        hl_dir = self.dirs[self.nref-1]
        for dir in self.dirs[0:self.nref-1]:
            for i in range(self.nscan):
                print dir,i
                for j in range(len(energy[dir][i])):
                    print j,energy[hl_dir][i][j],energy[dir][i][j]

    def print_compare_all(self,ref_energy,profiles):
        for dir in self.dirs[0:self.nref-1]:
            for fix in ["","BONDS","ANGLES"]:
                for i in range(self.nscan):
                    print "dir: {0} fix: {1} {2}".format(dir,fix,i)
                    for j in range(len(ref_energy[i])):
                        print j,ref_energy[i][j],profiles[dir][fix][i][j]

    def find_max_energy_index(self,log_file):
        gau = gaussian_log_file(log_file)
        vals, sinfo, en, xyz = gau.read_scan_info()
        maxE = en[0]; index = 0
        for j in range(1,len(xyz)):
            if (en[j] > maxE):
                maxE = en[j]
                index = j
        if (index==0): index = len(xyz)-1
        return index

    def get_copt_files(self):
        files = []
        mf = math_funs()
        for i in range(self.nscan):
            log_file = "./{0}/dihed_ref{1}.log".format(self.dirs[self.nref-2],i)
            gau = gaussian_log_file(log_file)
            vals, sinfo, en, xyz = gau.read_scan_info()
            maxE, index = mf.max_val(en)
            if (index==0): index = len(en)-1
            files.append("{0}/copt_{1}_{2}".format(self.dirs[nref-1],i,index))
        return files

    def get_SP_energies(self):
        LLenergy = {}
        for dir in self.dirs[0:self.nref-1]:
            print "getting SP energies for ",dir
            LLenergy[dir] = {}
            for fix in ["","BONDS","ANGLES"]:
                LLenergy[dir][fix] = []
                for i in range(self.nscan):
                    log_file = "./{0}/dihed_ref{1}{2}.log".format(dir,fix,i)
                    gau = gaussian_log_file(log_file)
                    vals, sinfo, en, xyz = gau.read_scan_info()
                    LLenergy[dir][fix].append([])
                    for j in range(len(xyz)):
                        gau = gaussian_log_file("{0}/SP_{1}_{2}_{3}.log".format(dir,fix,i,j))
                        LLenergy[dir][fix][-1].append(gau.read_energy())
                    for j in range(len(xyz)-1,-1,-1):
                        LLenergy[dir][fix][-1][j] = (LLenergy[dir][fix][-1][j] - LLenergy[dir][fix][-1][0])*627.51
        return LLenergy



class gaussian_oniom_file:
    def __init__(self,filename):
        """ Setup with g09 .log filename """
        self.filename = filename
        self.logfile = []
        self.elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]
        f = open(filename)
        for line in f:
            self.logfile.append(line)
        f.close()

    def get_coordinates(self):
        go = 0;        xyz = []        ; atom = []
        for line in self.logfile:
            m = re.search('orientation:',line)
            if (m):
                go=1
                xyz = []
            if (go>0): go = go + 1
            if (go>6):
                L = line.split()
                if (len(L)>4):
                    atom.append(self.elements[int(L[1])-1])
                    xyz.append([ float(L[3]),float(L[4]),float(L[5]) ])
                else:
                    go=0

        return atom, xyz

    def get_QM_energy(self):
        energy = None
        for line in self.logfile:
            m = re.search('high\s+system:\s+model energy:\s+([\d\-\.]+)',line)
            if (m):
                energy = m.group(1)

        return float(energy)
                
class gaussian_log_file:
    def __init__(self,filename=None):
        """ Setup with g09 .log filename """
        self.filename = filename
        self.logfile = []
        self.elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]
        self.atoms = []
        self.charge = -99
        self.mult = -99
        self.optjob = False
        self.xyz = []
        self.aname = []
        go = 0
        if (not filename):
            return
        f = open(filename)
        for line in f:
            if (re.match('\s*\#.+opt|OPT',line)):
                self.optjob = True
            m = re.search('Charge =\s+([\-\d]+)\s+Multiplicity =\s*([\-\d]+)',line)
            if (m):
                self.charge = int(m.group(1))
                self.mult = int(m.group(2))
            m = re.search('orientation:',line)
            if (go>0):  go = go + 1
            if (go>5):
                L = line.split()
                if (len(L)>4):
                    at = self.elements[int(L[1])-1]
                    self.atoms.append(Atom(at,map(float,L[3:6])))
                    self.aname.append(at)
                    self.xyz.append(map(float,L[3:6]))
                else:
                    go = 0
            if (m):
                go = 1
                del self.atoms
                self.atoms = []
                self.aname = []
                self.xyz = []                
            
            self.logfile.append(line)
        f.close()

    def get_charge(self):
        for line in self.logfile:
            m= re.search('\s*Charge =\s+([\d\-]+).+ltiplicity',line)
            if (m):       return int(m.group(1))

    def replace_xyz(self,xyz):
        natoms = len(self.atoms)

        for i in range(natoms):
            self.atoms[i].xyz = xyz[i]

    def rij(self,at1,at2):
        return self.atoms[at1].distance_to_atom(self.atoms[at2])

    def get_elements(self):
        go = 0; elements = []
        for line in self.logfile:
            m = re.search('orientation:',line)
            if (go>0): go = go + 1
            if (go>5):
                L = line.split()
                if (len(L)>4):
                    elements.append(int(L[1]))
                else:
                    return elements
            if (m): go = 1

    def read_esp_points(self):
        go = 0; charges = []
        for line in self.logfile:
            if (go>0): go = go + 1
            m = re.search('^\s*Charges from ESP fit.+RMS=\s+([\d\-\.]+)\s+RRMS',line)
            if (m):
                go = 1; charges = []
            if (go>3):
                L = line.split()
                if (len(L)!=3):
                    go =0
                else:
                    charges.append(float(L[2]))

        return charges

    def read_mk_charges(self):
        go = 0; charges = []
        for line in self.logfile:
            if (go>0): go = go + 1
            m = re.search('^\s*Mulliken atomic charges',line)
            if (m):
                go = 1; charges = []
            if (go>2):
                L = line.split()
                if (len(L)!=3):
                    go =0
                else:
                    charges.append(float(L[2]))

        return charges
            
    def read_scan_info(self):
        """ Read in the scanned structures and energies from the log file. """
        go = 0;        en = [];    xyzs = [];   vals = []; var = ""; sinfo=""
        for line in self.logfile:
            if (var):
                m = re.search('\!\s+' + var + '\s+.+\s+([\d\.\-]+)\s+[\-]*DE.+\!',line)
                if (m):
                    internc = m.group(1)
                    vals.append(internc)
            m = re.search('\s*\!\s+([\w\d]+)\s+([\d\.\-]+)\s+Scan',line)
            if (m):
                var = m.group(1)
                internc = m.group(2)
            m = re.search('\!\s+([\w\d]+)\s+\w+\((.+)\)\s+([\d\.\-]+)\s+Scan',line)
            if (m):
                var = m.group(1)
                sinfo = m.group(2).split(',')
                internc = m.group(3)
            m = re.search('orientation:',line)
            if (m):
                go=1
                xyz = []
            if (go>0): go = go + 1
            if (go>6):
                L = line.split()
                if (len(L)>4):
                    xyz.append([self.elements[int(L[1])-1], float(L[3]),float(L[4]),float(L[5]) ])
                else:
                    go=0
            m = re.search('SCF Done.+\=\s+([\+E\d\.\-]+)',line)
            m2 = re.match(' Energy=\s+([\d\.\-]+)\s+NIter=',line)
            m3 = re.search('EUMP2\s*=\s+([\+DE\d\.\-]+)',line)
            if (m):            energy = float(m.group(1))
            if (m2):           energy = float(m2.group(1))
            if (m3):
                num = m3.group(1).replace("D","E")
                energy = float(num)

            m = re.search('Optimization completed',line)
#            m2 = re.search('Optimization stopped',line)
            m3 = re.search('Number of steps exceeded',line)
            if (m or m3):
                print "line: ", line
                en.append(energy)
                xx = deepcopy(xyz)
                xyzs.append(xx)
                
        return vals,sinfo,en,xyzs

    def read_energy(self,QMonly=False):
        energy = 0
        oniom = False
        for line in self.logfile:
            if (QMonly):
                m = re.search("high\s+system:\s+model energy:\s+([\-\d\.]+)",line)
            else:
                m = re.search(" ONIOM: extrapolated energy =\s+([\-\d\.]+)",line)            
            m2 = re.search('SCF Done.+\=\s+([\d\.\-]+)',line)
            m3 = re.match(' Energy=\s+([\d\.\-E]+)\s+NIter=',line)
            if (m):
                energy = float(m.group(1))
                oniom = True
            if (m2 and not oniom): energy = float(m2.group(1))
            if (m3 and not oniom): energy = float(m3.group(1))            

        return energy

    def read_gibbs_energy(self):
        energy = 0
        for line in self.logfile:
            m = re.search('Sum of electronic and thermal Free Energies\=\s+([\d\.\-]+)',line)
            if (m):
                energy = float(m.group(1))

        return energy

    def read_energies(self):
        energy = []
        
        for line in self.logfile:
            m2 = re.search('SCF Done.+\=\s+([\d\.\-]+)',line)
            m3 = re.match(' Energy=\s+([\d\.\-E]+)\s+NIter=',line)
            if (m2): energy.append(float(m2.group(1)))
            if (m3): energy.append(float(m3.group(1)))

        return energy

    def read_MM_diff(self):
        e1 = 0;        e2 = 0
        for line in self.logfile:
            m = re.search("low\s+system:\s+model energy:\s+([\-\d\.]+)",line)
            m2 = re.search("low\s+system:\s+real\s+energy:\s+([\-\d\.]+)",line)
            if (m):                e1 = float(m.group(1))
            if (m2):                e2 = float(m2.group(1))

        return e2 - e1

    def read_frozen(self):
        for line in self.logfile:                
            m = re.search('\!\s+([\w\d]+)\s+\w+\((.+)\)\s+([\d\.\-]+)\s+Frozen',line)
            if (m):
                var = m.group(1)
                sinfo = m.group(2).split(',')
                return m.group(3)
            
    def read_xyz(self):
        """ Read in the structures and energies from the log file from a scan. """
        go = 0; xyz=[]
        for line in self.logfile:
            m = re.search('nput\s+orientation:',line)
            m2 = re.search('ard\s+orientation:',line)
            if (m2 or m):
                go = 1
                xyz = []
            if (go>0): go = go + 1
            if (go>6):
                if (re.search('\-{6}',line)):
                    go =0
                else:
                    L = line.split()
                    xyz.append([self.elements[int(L[1])-1],float(L[3]),float(L[4]),float(L[5])])
                    
        return xyz

    def write_xyzfile(self,xyz,filename,comment=""):
        f = open(filename,'w')
        f.write(str(len(xyz)) + "\n{0}\n".format(comment))
        for xx in xyz:
            f.write("{0} {1} {2} {3}\n".format(xx[0],xx[1],xx[2],xx[3]))

    def write_com_file(self,header,out_file,tailer=None):
        f = open(out_file,'w')
        f.write(header)
        for i in range(len(self.xyz)):
            f.write("{0} {1} {2} {3}\n".format(self.aname[i],self.xyz[i][0],self.xyz[i][1],self.xyz[i][2]))
        if (tailer):
            f.write("\n")
            f.write(tailer)
        f.write("\n\n")
        f.close()

    def write_com(self,header_file,out_file,tail_file=None,tail_string=None):
        """ Write out a sequence of Gaussian com files. """

        # Get the head and tail information.
        f = open(header_file)
        header = f.read()
        f.close()
        if (tail_file):
            f = open(tail_file)
            tailer = f.read()
            f.close()

        f = open(out_file,'w')
        f.write(header)
        f.write("\ncomment\n\n")
        f.write("{0} {1}\n".format(self.charge,self.mult))
        for index in range(len(self.atoms)):
            f.write(self.atoms[index].xyz_str() + "\n")

        if (tail_file): f.write(tailer)
        if (tail_string): f.write(tail_string)
        f.write("\n\n")
        f.close()
    

    def write_com_files(self,xyz,header_file,tail_file,sinfo,vals):
        """ Write out a sequence of Gaussian com files. """

        # Get the head information.
        f = open(header_file)
        header = f.read()
        f.close()
        f = open(tail_file)
        tailer = f.read()
        f.close()        
        for i in range(len(xyz)):
            f = open('file' + str(i) + '.com','w')
            f.write(header)
            for j in range(len(xyz[i])):
                f.write(" ".join(map(str,xyz[i][j])) + "\n")
            f.write("\n" + " ".join(sinfo) + " " + str(vals[i]) + " F\n\n")
            f.write(tailer)            
            f.close()

    def distances_btw_atoms(self,xyzs,a1,a2):
        dist = []
        for i in range(len(xyzs)):
            vec = array(xyzs[i][a1][1:4]) - array(xyzs[i][a2][1:4])
            dist.append(sqrt(dot(vec,vec)))
        return dist

    def relative_barrier(self,en,index=0):
        val = en[index]
        for i in range(len(en)):
            en[i] = (en[i] - val)*627.51
        return en

    def write_xyzs(self,xyz,filename,useatoms=False):
        f = open(filename,'w')
        if (useatoms):
            for i in range(len(xyz)):
                f.write(str(len(xyz[i])) + "\n\n")
                for j in range(len(xyz[i])):
                    f.write('{0} {1} {2} {3}\n'.format(self.atoms[i].aname,xyz[i][j][0],xyz[i][j][1],xyz[i][j][2]))
        else:
            for i in range(len(xyz)):
                f.write(str(len(xyz[i])) + "\n\n")
                for j in range(len(xyz[i])):
                    f.write('{0} {1} {2} {3}\n'.format(xyz[i][j][0],xyz[i][j][1],xyz[i][j][2],xyz[i][j][3]))
        f.close()

    def read_g03_Prop(self):
        fit = []; go = 0 
        for line in self.logfile:        
            m = re.search('Electrostatic Properties \(Atomic Units\)',line)
            m2 = re.search('^\s*\d+\s+Atom\s+([\d\-\.]+)',line)
            m3 = re.search('^\s*[\*\d]+\s+([\d\-\.]+)',line)            
            if (m): go = 1
            if (go==1 and m2): go = 2
            if (m3 and go==2):
                fit.append(float(m3.group(1)))

        return fit

    def read_g03_ESP_centers(self):
        """ Read in the ESP centers from the g03 log file """
        atomic_centers = []
        esp_fit_centers = []
        fit = []
        for line in self.logfile:
            m = re.search('Atomic Center\s+\d+ is at\s+([\d\.\-]+)\s+([\d\.\-]+)\s+([\d\.\-]+)',line)
            m2 = re.search('ESP Fit Center\s+[\*\d]+ is at\s+([\d\.\-]+)\s+([\d\.\-]+)\s+([\d\.\-]+)',line)
            m3 = re.search('Fit    \s*([\d\.\-]+)',line)
            if m: atomic_centers.append(map(float,m.groups()))
            if m2: esp_fit_centers.append(map(float,m2.groups()))
            if m3:
                fit.append(float(m3.group(1)))
        print len(fit),len(esp_fit_centers)
        return atomic_centers,esp_fit_centers,fit

    def print_ESP_points(self,ac,esp):
        f = open("esp.xyz",'w')
        f.write(str(len(ac)+len(esp))+"\n\n")
        for i in range(len(ac)):
            f.write("C " + " ".join(map(str,ac[i])) + "\n")
        for i in range(len(esp)):
            f.write("P " + " ".join(map(str,esp[i])) + "\n")
        f.close()

    def associate_centers_and_trim(self,ac,efc,fit,filename):
        """ Associate each esp point with a particular center based on distance. """
        f = open(filename)
        residue = []
        new_ac = []
        new_efc = []
        new_fit = []
        i=0
        for line in f:
            m = re.match('True',line)
            residue.append(m!=None)
            if (m):
                new_ac.append(ac[i])
                i = i + 1

        print residue
        for j in range(len(efc)):
            efc[j] = array(efc[j])
        for j in range(len(ac)):
            ac[j] = array(ac[j])
        
        for j in range(len(efc)):
            mindist = 10
            if (j %1000==0): print j
            for i in range(len(ac)):
                vec = efc[j] - ac[i]
                dist = linalg.norm(vec)
                if (dist<mindist):
                    mindist = dist
                    cent = i
#            print mindist,cent,efc[j],ac[cent]
            if (residue[cent]):
                new_efc.append(efc[j])
                new_fit.append(fit[j])
        print len(new_efc),len(efc)

        return new_ac, new_efc,new_fit
        
    def write_ESP(self,ac,efc,fit):
        unit=0.529177249
        f = open("a",'w')
        for i in range(len(ac)):
            f.write('                {0: 16.6E}{1: 16.6E}{2: 16.6E}\n'.format(ac[i][0],ac[i][1],ac[i][2]))
        f.close()
        f = open("b",'w')        
        for i in range(len(efc)):
            f.write('{0: 16.6E}{1: 16.6E}{2: 16.6E}{3: 16.6E}\n'.format(fit[i],efc[i][0],efc[i][1],efc[i][2]))
        f.close()
        f = open("num.dat",'w')
        f.write('{0} {1}'.format(len(ac),len(efc)))
        f.close()


class Molecule:
    """ Contains all the properties of a molecule """
    def __init__(self,name="",xyz=[]):
        self.name = name
        self.atoms=[]
        self.bond_list=[]
        self.bond_type=[]
        self.bond_matrix = []
        if (xyz):
            for xx in xyz:
                self.atoms.append(Atom(xx[0],map(float,xx[1:4])))

    def closest_atom(self,at2):
        low = -1; mindist = 99
        for i,at in enumerate(self.atoms):
            dist = at2.distance_to_atom(at)
            if (dist<mindist and at2 != at):
                mindist = dist
                low = i
        return low

    def move_struct(self,atoms,val,charge,mult):
#        print atoms,val
        head = ["#am1 geom=modredundant\n\ncomment\n\n","{0} {0}\n".format(charge,mult)]

        tail = [" ".join(map(lambda x: str(x+1),atoms)) + " {} F".format(val)]
        self.write_com("tmp.com",head,tail)
        os.system("g03 tmp.com")
        self.set_xyz_from_log_file("tmp.log")

    def set_xyz_from_log_file(self,log_file):
        gl = gaussian_log_file(log_file)
        xyz = gl.read_xyz()
        for i in range(len(self.atoms)):
            self.atoms[i].xyz = map(float,xyz[i][1:])

    def write_com(self,filename,head,tail):
        """ Writes out a .com file """
        f = open(filename,'w')

        # Write the head.
        for line in head:
            f.write(line)

        # Write the xyz coordinates.
        for index in range(len(self.atoms)):
            f.write(self.atoms[index].xyz_str() + "\n")

        # Write the tail.
        f.write("\n")
        for line in tail:
            f.write(line)
            
        f.write("\n\n")
        f.close()
        

    def rij(self,at1,at2):
        return self.atoms[at1].distance_to_atom(self.atoms[at2])
    
    def unique_atoms(self):
        uatoms = []
        for at in self.atoms:
            if (not at.atom in uatoms):
                uatoms.append(at.atom)

        return uatoms

    def heavy_connected(self,start,depth):
        con = [ start ]
        newcon = [ start ]
        for i in range(depth):
            new1 = copy(newcon)
            newcon = []
            for c1 in new1:
                for cc in self.bond_matrix[c1]:
                    if (not cc in con and self.atoms[cc].elnum>1):
                        newcon.append(cc)
                        con.append(cc)

        return con[1:]
    
    def copy_mol(self,other_mol):
        self.atoms = copy(other_mol.atoms)        
        
    def copy_over(self,other,elements):
        self.name = other.name
        for index in range(len(elements)):
            self.atoms.append(other.atoms[elements[index]])

    def get_internal(self,index):
        if (len(index)==2): return self.bond_length(index[0],index[1])
        elif (len(index)==3): return self.bond_angle(index[0],index[1],index[2])
        elif (len(index)==4): return self.dihed_angle(index[0],index[1],index[2],index[3])
        else:
            print "Unknown internal coordinates",index; sys.exit(0)

    def bond_length(self,i,j):
        x1 = array(self.atoms[i].xyz)
        x2 = array(self.atoms[j].xyz)
        vec = x2 - x1

        return linalg.norm(vec)

    def bond_angle(self,i,j,k):
        x1 = array(self.atoms[i].xyz)
        x2 = array(self.atoms[j].xyz)
        x3 = array(self.atoms[k].xyz)

        a = x1 - x2
        a = a / linalg.norm(a)
        b = x3 - x2
        b = b / linalg.norm(b)

        out = arccos(clip(dot(a,b),-1,1))*57.2957795
        if (out<0): out = out + 180
        
        return out

    def dihed_angle(self,i,j,k,l):
        x1 = array(self.atoms[i].xyz)
        x2 = array(self.atoms[j].xyz)
        x3 = array(self.atoms[k].xyz)
        x4 = array(self.atoms[l].xyz)
        a = x1 - x2
        b = x3 - x2
        c = x4 - x3
        b = b / linalg.norm(b)
        tmp = b.copy()
        tmp = tmp*dot(a, b)
        a = a - tmp
        tmp = b.copy()
        tmp = tmp*dot(c, b)
        c = c - tmp
        a = a/linalg.norm(a)
        c = c/linalg.norm(c)

        sign = 1-(linalg.det([a, b, c]) > 0)*2

        return sign*arccos(clip(dot(a, c),-1,1))*57.2957795

    def merge(self,other,num_file):
        # Get numbering for excluded atoms.
        f = open(num_file,'r')
        tmp = f.readline(); excludeA = map(int,tmp.split())
        tmp = f.readline(); excludeB = map(int,tmp.split())
        f.close()

        mol3 = Molecule()
        
        # Put non-excluded elements in the new molecule.
        for i in range(len(self.atoms)):
            if (not i in excludeA):
                mol3.atoms.append(self.atoms[i])
        for i in range(len(other.atoms)):
            if (not i in excludeB):
                mol3.atoms.append(other.atoms[i])

        # Add in residue information.
        for at in mol3.atoms:
            at.resid = 1
            at.resname = "pSR"
            
        return mol3

    def get_charges(self,file):
        f = open(file,'r')
        line = f.read()
        L = map(float,line.split())
        return L

    def merge_charges(self,cfile1,cfile2,num_file):
        # Get the charges
        c1 = self.get_charges(cfile1)
        c2 = self.get_charges(cfile2)

        # Get the atoms to exclude.
        f = open(num_file,'r')
        line = f.readline(); exclude1 = map(int,line.split())
        line = f.readline(); exclude2 = map(int,line.split())
        f.close()

        # Change the charges
        nn = 0 
        for i in range(len(c1)):
            if (not i in exclude1):
                self.atoms[nn].charge = c1[i]
                nn = nn + 1

#        print exclude1
#        print nn,len(c1)
#        sys.exit(0)

        for i in range(len(c2)):
            if (not i in exclude2):
                self.atoms[nn].charge = c2[i]
                nn = nn + 1
            

    def read_mol2(self,filename):
        """ Read in a .mol2 file """
        f = open(filename,'r')
        start=0
        natom=0
        for line in f:
            if re.search('SUBSTR',line): start = 0
            if (start==2):
                L = line.split()
                if (len(L)>3):
                    self.bond_list.append(map(int,L[1:3]))
                    self.bond_type.append(L[3])
            if re.search('BOND',line): start = 2
            if (start==1):
                L = line.split()
                if (len(L)>7):
                    self.atoms.append(Atom(L[1],map(float,L[2:5]),L[5],int(L[6]),L[7],float(L[8])))
            if re.search('ATOM',line): start = 1
        f.close()

    def protonate(self,name):
        for at in (self.atoms):
            if (at.aname.strip() == name.strip()):
                ind = self.closest_atom(at)
                vec = array(at.xyz) - array(self.atoms[ind].xyz)
                vec = vec / sqrt( dot(vec,vec) )
                at2 = self.atoms[-1]
                self.atoms.append(Atom(at2.aname,array(at.xyz)+vec,at2.atype,at2.resid,at2.resname,at2.charge,at2.chain))
                m = re.search('(\d+)',at2.aname)
                self.atoms[-1].aname="H{0}".format(int(m.group(1))+1)
                return

    def read_pdb(self,filename):
        """ Read in a .mol2 file """
        f = open(filename,'r')
#ATOM    982  CA  SER A 219     147.242  93.314   2.908  1.00 46.42           C
#    def __init__(self,aname,xyz,atype="",resid=1,resname="ZNC",charge=0.0,chain="A"):
        for line in f:
            if (re.match('ATOM',line) or re.match('HETATM',line)):
                m = re.match('(.{6})(.{5})(.{5})(.{4})(.{2})(.{4})(.+)',line)
                L = m.group(7).split()
                self.atoms.append(Atom(m.group(3).strip(),map(float,L[0:3]),m.group(3).strip(),int(m.group(6)),m.group(4).strip(),0.0,m.group(5).strip()))


        f.close()

    def atoms_from_resid(self,name):
        out = []
        for i,at in enumerate(self.atoms):
            if (at.resname==name):
                out.append(i)
        return out

    def add_caps(self):
        resid = self.atoms[0].resid
        chain = self.atoms[0].chain
        change = False;res=0
        for i,at in enumerate(self.atoms):
            at.resid+=res
            if (at.aname=="C"):
                # Find the closest atoms
                N=-1; CA=-1; O=-1
                for j,at2 in enumerate(self.atoms):
                    dist = self.bond_length(i,j)
                    if (dist<1.7):
                        if (at2.aname=="CA"): CA=j
                        if (at2.aname=="O"): O=j
                        if (at2.aname=="N"): N=j

                print "check: ",N,CA,O,at.resid,at.chain
                if (N==-1 and CA>-1 and O>-1):
                    print "adding"
                    Oxyz = array(self.atoms[O].xyz)
                    CAxyz = array(self.atoms[CA].xyz)
                    Cxyz = array(at.xyz)
                    vec = (CAxyz - Cxyz) + (Oxyz - Cxyz)
                    newxyz = Cxyz - 1.4*vec/linalg.norm(vec)
                    change = True
                    
            if (resid!=at.resid or chain!=at.chain):
                if (change):
                    print "adding..."
                    self.atoms.insert(i,Atom("N",map(float,newxyz[0:3]),"N",at.resid+res,"NME",0.0,at.chain))
                    res+=1
                    at.resid+=1
                    change = False
                resid = at.resid
                chain = at.chain
            print "check ", at.aname,self.atoms[i].aname,at.resid
                    
                

    def trim_res_away_from_res(self,name,radius):
        atoms = self.atoms_from_resid(name)
        xyz = array( [ 0.0, 0.0, 0.0 ] )
        for i in atoms:
            xyz += self.atoms[i].xyz
        xyz = xyz / len(atoms)
        resid = self.atoms[0].resid
        chain = self.atoms[0].chain
        mindist = 1000
        index = []

        for i,at in enumerate(self.atoms):
            vec = xyz - array(at.xyz)
            dist = linalg.norm(vec)
            if (dist<mindist):
                mindist = dist
            if (at.resid!=resid or at.chain!=chain):
                print mindist,radius, chain,resid
                if (mindist>radius):
                    index.append([chain,resid])
#                print chain, resid,i
                chain = at.chain
                resid = at.resid
                mindist = 1000

        newatoms = []
        for at in self.atoms:
            if (not [at.chain,at.resid] in index):
                newatoms.append(at)

        self.atoms = newatoms


    # List the residues beyond a certain distance.
    def res_away_from_res(self,name,radius):
        atoms = self.atoms_from_resid(name)
        xyz = array( [ 0.0, 0.0, 0.0 ] )
        for i in atoms:
            xyz += self.atoms[i].xyz
        xyz = xyz / len(atoms)
        resid = self.atoms[0].resid
        chain = self.atoms[0].chain
        mindist = 1000
        out = ""
        num = 1

        for i,at in enumerate(self.atoms):
            vec = xyz - array(at.xyz)
            dist = linalg.norm(vec)
            if (dist<mindist):
                mindist = dist
            if (at.resid!=resid or at.chain!=chain):
                print mindist,radius, chain,resid
                if (mindist>radius):
                    out = out + "(:{0}-{0}) &".format(num)
                chain = at.chain
                resid = at.resid
                num += 1
                mindist = 1000

        return out

        
    def read_mol(self,filename):
        """ Read in a .mol2 file """
        f = open(filename,'r')
        f.readline();        f.readline();        f.readline()
        line = f.readline()                
        L = line.split(); natoms = int(L[0]); nbonds = int(L[1])

        self.atoms = []
        for i in range(natoms):
            line = f.readline();            L = line.split()
            self.atoms.append(Atom(L[3],map(float,L[0:3])))
        for i in range(nbonds):
            line = f.readline();            L = line.split()            
            self.bond_list.append(map(int,L[0:2]))
            self.bond_type.append(L[2])
        f.close()

    def build_bond_angle_dihed_list(self):
        bonds = []
        angles = []
        diheds = []

        # Build bond matrix
        natoms = len(self.atoms)
        self.build_bond_matrix()

        # Get all unique bonds, angles and dihedrals.
        for i in range(natoms):
            br = self.bond_matrix[i]
            for j in br:
                if (j!=i):
                    if (j>i): bonds.append([i,j])
                    br2 = self.bond_matrix[j]
                    for k in br2:
                        if (k!=j and k!=i):
                            if(k>i): angles.append([i,j,k])
                            br3 = self.bond_matrix[k]
                            for l in br3:
                                if (l!=k and l!=j and l > i):
                                    diheds.append([i,j,k,l])

        return bonds,angles,diheds

    def read_xyz(self,filename):
        """ Read in a .xyz file """
        f = open(filename,'r')
        natom = int(f.readline()) + 0
        self.atoms = []
        f.readline()
        for i in range(natom):
            line = f.readline()
            L = line.split()
            self.atoms.append(Atom(L[0],map(float,L[1:4])))
        f.close()

    def read_replace_xyz(self,filename):
        """ Read in a .xyz file """
        f = open(filename,'r')
        natom = int(f.readline()) + 0
        f.readline()
        for i in range(natom):
            line = f.readline()
            L = line.split()
            self.atoms[i].xyz = map(float,L[1:4])
        f.close()

    def read_com(self,filename):
        """ Read in a Gaussian .com file """
        com_head = []
        com_tail = []
        f = open(filename,'r')
        xyz=0
        for line in f:
            L = line.split()
            if (xyz==0):
                com_head.append(line)
                if re.match('[\-\d]+\s+[\-\d]+\s*$',line):
                    xyz = 1
                    charge = int(L[0])
            if (xyz==1 and len(L)>3):
                self.atoms.append(Atom(L[0],map(float,L[1:4])))
            elif (xyz==1 and len(L)<2):
                xyz=2
            elif (xyz==2):
                com_tail.append(line)

        f.close()
        
    def total_charge(self):
        sum = 0.0
        for atom in self.atoms:
            sum = sum + float(atom.charge)
        return sum
    def find_NCconnect(self):
        connect = [0,0]
        index = 0
        for atom in self.atoms:
            index = index + 1
            if (atom.aname=="N"):
                connect[0] = index
                print "check: ", atom.aname
            if (atom.aname=="C"):
                connect[1] = index
                print "check: ", atom.aname
            
        return connect

    def write_mol2(self,filename):
        """ Writes out a .mol2 file """
        f = open(filename,'w')
        f.write("@<TRIPOS>MOLECULE\n")
        f.write(self.atoms[0].resname + "\n")
        f.write('{0:5d} {1:6d}     1     0     0\n'.format(len(self.atoms),len(self.bond_list)))
        f.write("SMALL\n\n")
        f.write("@<TRIPOS>ATOM\n")

        for index in range(len(self.atoms)):
            f.write(self.atoms[index].mol2_str(index) + "\n")
        f.write("@<TRIPOS>BOND\n")
        for index in range(len(self.bond_list)):
            f.write('  {0:4d} {1:4d} {2:4d} {3}\n'.format(index+1,self.bond_list[index][0],self.bond_list[index][1],self.bond_type[index]))
        f.write('@<TRIPOS>SUBSTRUCTURE\n')
        f.write('1 {0}         1 TEMP              0 ****  ****    0 ROOT            \n'.format(self.atoms[0].resname))

        f.close()
    def write_xyz(self,filename):
        """ Writes out a .xyz file """
        f = open(filename,'w')
        f.write('{0}\n\n'.format(len(self.atoms)))
        for index in range(len(self.atoms)):
            f.write(self.atoms[index].xyz_str() + "\n")
        f.close()
    def write_xyzs(self,xyzs,filename,comment=""):
        """ Writes out a .xyz file """
        f = open(filename,'w')
        for i in range(len(xyzs)):
            f.write('{0}\n{1}\n'.format(len(self.atoms),comment))
            self.replace_xyz(xyzs[i])
            for index in range(len(self.atoms)):
                f.write(self.atoms[index].xyz_str() + "\n")
        f.close()
    def write_pdb(self,filename):
        """ Writes out a .pdb file """
        f = open(filename,'w')
        for index in range(len(self.atoms)):
            f.write(self.atoms[index].pdb_str(index) + "\n")
        f.close()
    def write_pack_pdb(self,filename,num):
        """ Writes out a .pdb file """
        mf = math_funs()        
        hh = [[-9999,9999],[-9999,9999],[-9999,9999]]; diff = []
        natoms = len(self.atoms)
        for i in range(3):
            for at in self.atoms:
                hh[i][0] = max(hh[i][0],at.xyz[i])
                hh[i][1] = min(hh[i][1],at.xyz[i])
            diff.append(hh[i][0] - hh[i][1] + 3.0)
        ind = [0, 0, 0]
        xyz = []
        for i in range(natoms):
            xyz.append(copy(self.atoms[i].xyz))
        f = open(filename,'w')
        for i in range((num+1)**3):
            for k,at in enumerate(self.atoms):
                for j in range(3):
                    at.xyz[j] = xyz[k][j] + ind[j]*diff[j]
                f.write(at.pdb_str(k+natoms*i) + "\n")
                at.resid += 1
            mf.augment_array(ind,num)
            
        f.close()
    def read_in_charges(self,filename):
        """ Read in charges from a file """
        f = open(filename,'r')
        line = f.read()
        L = map(float,line.split())
        for i in range(len(L)):
            self.atoms[i].charge = L[i]

    def remove_atom(self,index):
        """ Remove atoms indexed from 0 """
        del self.atoms[index-1]

        new_bl=[]
        new_bt=[]

        for i in range(len(self.bond_list)):
            if self.bond_list[i][0] == index or self.bond_list[i][1] == index:
                pass
            else:
                if (self.bond_list[i][0] > index): self.bond_list[i][0] = self.bond_list[i][0] - 1
                if (self.bond_list[i][1] > index): self.bond_list[i][1] = self.bond_list[i][1] - 1
                new_bl.append(self.bond_list[i])
                new_bt.append(self.bond_type[i])
        self.bond_list = new_bl
        self.bond_type = new_bt
        print self.bond_list

    def build_bond_matrix(self):
        """ Setups a bond matrix (natoms x natoms) """
        natoms = len(self.atoms)

        for i in range(natoms):
            self.bond_matrix.append([])

        for i,j in self.bond_list:
            self.bond_matrix[i-1].append(j-1)
            self.bond_matrix[j-1].append(i-1)

    def reorder_atoms(self,index,index2):
        """ Switch atoms index and index2 (starting from 1)"""
        # Switch the atom information.
        atom = self.atoms[index2-1]
        self.atoms[index2-1] = self.atoms[index-1]
        self.atoms[index-1] = atom

        # Change the bonding pattern.
        for i in range(len(self.bond_list)):
            for j in range(2):
                if self.bond_list[i][j] == index:
                    self.bond_list[i][j] = index2
                elif self.bond_list[i][j] == index2:
                    self.bond_list[i][j] = index

    def add_charges_based_norm(self,other_mol,tol):
        for ind in range(len(self.atoms)):
            maxval = tol
            index = -1
            for ind2 in range(len(other_mol.atoms)):
                a = array(self.atoms[ind].xyz)
                b = array(other_mol.atoms[ind2].xyz)
                norm = linalg.norm((a-b),ord=2)
                if (norm<maxval):
                    index = ind2
                    maxval = norm
            if (index>-1):
                self.atoms[ind].charge = other_mol.atoms[index].charge

    def copy_xyz(self,other_mol):
        for ind in range(min(len(self.atoms),len(other_mol.atoms))):
            for i in range(3):
                self.atoms[ind].xyz[i] = other_mol.atoms[ind].xyz[i]

    def replace_xyz(self,xyz,extra=[]):
        for i in range(min(len(xyz),len(self.atoms))):
            if (len(xyz[0])==3):
                for j in range(3):
                    self.atoms[i].xyz[j] = float(xyz[i][j])
            else:
                for j in range(1,4):
                    self.atoms[i].xyz[j-1] = float(xyz[i][j])
        if (len(extra)>0):
            for i,ex in enumerate(extra):
                for j in range(3):
                    self.atoms[len(xyz)+i].xyz[j] = float(xyz[ex-1][j])+0.001

    def copy_charges(self,other_mol):
        for ind in range(len(self.atoms)):
            if (other_mol is Molecule):
                self.atoms[ind].charge = other_mol.atoms[ind].charge
            else:
                self.atoms[ind].charge = other_mol[ind]
    def read_and_assign_charges(self,other_mol,filename):
        f = open(filename,'r') # read from .chg file.
        charges = []
        for line in f:
            charges.extend(map(float,line.split()))
        sum =0
        done = []
        for x in range(len(charges)):
            sum += charges[x]
            done.append(0)
        print "Total charge: ",sum
        sum = 0
        for ind in range(len(self.atoms)):
            minnorm = 1.2
            index = -1
            for ind2 in range(len(other_mol.atoms)):
                a = array(self.atoms[ind].xyz)
                b = array(other_mol.atoms[ind2].xyz)
                norm = linalg.norm((a-b),ord=2)
                if (norm<minnorm and self.atoms[ind].atom == other_mol.atoms[ind2].atom):
                    minnorm = norm
                    index = ind2
            if (index>-1):
                self.atoms[ind].charge=charges[index]
                done[index] = 1
                sum = sum + charges[index]
            else:
                self.atoms[ind].charge = 9.9999
        print done
        print "Sum: ", sum

    def assign_charges(self,charges):
        for ind in range(len(self.atoms)):
            self.atoms[ind].charge=charges[ind]

    def read_charges(self,filename):
        f = open(filename,'r') # read from .chg file.
        charges = []
        for line in f:
            charges.extend(map(float,line.split()))
        f.close()
        self.assign_charges(charges)
            
    def add_bonds(self,index,cutoff):
        a = array(self.atoms[index].xyz)
        for ind in range(len(self.atoms)):
            if (index==ind): continue
            b = array(self.atoms[ind].xyz)
            norm = linalg.norm((a-b),ord=2)
            if (norm < cutoff):
                print "norm: ", norm
                present=0 # check to make sure it isn't already there.
                for i in range(len(self.bond_list)):
                    if ((self.bond_list[i][0] == index+1 and self.bond_list[i][1] == i+1) or (self.bond_list[i][0] == i+1 and self.bond_list[i][1] == index+1)): present=1
                if (present==0):
                    self.bond_list.append([index+1, ind+1])
                    self.bond_type.append(1)

    def change_resnames(self,resid,letter):
        for i in range(len(self.atoms)):
            atom = self.atoms[i]
            if (atom.resid==resid):
                self.atoms[i].aname = atom.aname[0] + letter + atom.aname[2:]

    def write_xyz_esp(self,esp,filename):
        f = open(filename,'w')
        esp0 = esp.split("\n")
        f.write("{0:d}\n\n".format(len(self.atoms) + len(esp0)))

        for at in self.atoms:
            f.write("{0} {1:.4f} {2:.4f} {3:.4f}\n".format(at.aname,at.xyz[0],at.xyz[1],at.xyz[2]))
            
        for es in esp0[:-1]:
            f.write("He  " + es + "\n")
        
        f.close()
        
    def write_g09_com(self,filename,header,tailer,mult=1,charge=0.5):
        """ Write out a sequence of Gaussian com files. """

        # Get the head information.
        if (charge==0.5): charge = int(round(self.total_charge()))
        f = open(filename + '.com','w')
        f.write(header + "\n\ncomment\n\n")
        f.write("{0:d} {1:d}\n".format(charge,mult))
        for j in range(len(self.atoms)):
            at = self.atoms[j]
            f.write("{0} {1:.4f} {2:.4f} {3:.4f}\n".format(at.aname,at.xyz[0],at.xyz[1],at.xyz[2]))

        f.write("\n" + tailer + "\n")
        f.close()

    def align_to_mol(self,mol2,filenum):
        """ Align this molecule to mol2 using the atom numbering in nums. """
        # Used algorithm from: http://is.muni.cz/www/140435/thesis_20071128.pdf
        # Read in the atom numbers for atoms to align
        f = open(filenum,'r')
        nums = []
        for line in f:
            L = line.split()
            nums.append(map(int,L))
        f.close()
        
        # Get xyz coordinates.
        N = len(nums[0])
        A = zeros((3,N))
        B = zeros((3,N))
        A_cm = zeros(3)
        B_cm = zeros(3)        

        # Read in the numbered coordinates.
        for i in range(N):
            for j in range(3):
                A[j][i] = self.atoms[nums[0][i]].xyz[j]
                B[j][i] = mol2.atoms[nums[1][i]].xyz[j]
                A_cm[j] = A_cm[j] + A[j][i]
                B_cm[j] = B_cm[j] + B[j][i]                

        # Set the center of mass.
        A_cm = A_cm / N
        B_cm = B_cm / N

        # Shift to the center of mass.
        for i in range(N):
            A[:,i] = A[:,i] - A_cm
            B[:,i] = B[:,i] - B_cm

        # Figure out the matrix YY
        CC = dot(A,B.transpose())

        # Do the SVD
        U,s,Vh = linalg.svd(CC)
        R = dot(Vh.transpose(),U.transpose())

        # Put the coordinates back in two molecules.
        for at in self.atoms:
            at.xyz = dot(R,array(at.xyz) - A_cm)
        for at in mol2.atoms:
            at.xyz = array(at.xyz) - B_cm

class amber_Molecule(Molecule):
    def __init__(self,prefix="tmp"):
        Molecule.__init__(self,prefix)
        self.prefix = prefix

    def match_EPs(self):
        atoms = []
        for i,at in enumerate(self.atoms):
            if (at.atom=="EP"):
                j = self.closest_atom(at)
                atoms.append([i,j])

        return atoms

    def ATparameterize(self):
        self.write_pdb(self.prefix + ".pdb")
        os.system("antechamber -i {0}.pdb -fi pdb -o {0}.mol2 -fo mol2".format(self.prefix))
        os.system("rm ANTECH* ATOMTYP*")
        os.system("parmchk -i {0}.mol2 -a 'Y' -f mol2 -o all.frcmod".format(self.prefix))
        os.system("parmchk -i {0}.mol2  -f mol2 -o {0}.frcmod".format(self.prefix))        
        f = open("leap.in",'w')
        f.write("source leaprc.gaff\n")
        f.write("PER = loadmol2 {0}.mol2\nsaveoff PER {0}.lib\nsaveamberparm PER {0}.prmtop {0}.inpcrd\n".format(self.prefix))
        f.write("quit")
        f.close()
        os.system("tleap -f leap.in > leap.out")

    def ATparameterize_with_charges(self):
        self.write_pdb(self.prefix + ".pdb")
        os.system("antechamber -i {0}.pdb -fi pdb -o {0}.mol2 -fo mol2 -c bcc -s 2".format(self.prefix))
        os.system("rm ANTECH* ATOMTYP*")
        os.system("parmchk -i {0}.mol2  -f mol2 -o {0}.frcmod".format(self.prefix))
        f = open("leap.in",'w')
        f.write("source leaprc.gaff\n")
        f.write("PER = loadmol2 {0}.mol2\nsaveoff PER {0}.lib\nsaveamberparm PER {0}.prmtop {0}.inpcrd\n".format(self.prefix))
        f.write("quit")
        f.close()
        os.system("tleap -f leap.in > leap.out")

    def write_inpcrd(self,filename=None):
        if (not filename): filename = self.prefix + ".inpcrd"
        natoms = len(self.atoms)
        f = open(filename,'w')
        f.write("ZNC\n")
        f.write("{0:6d}\n".format(natoms))
        for i in range(natoms):
            f.write("{0: 12.7f}{1: 12.7f}{2: 12.7f}".format(*self.atoms[i].xyz))
            if (i % 2==1): f.write("\n")
        f.close()

    def run_AMBER(self,prmtop,in_file,inpcrd_file,output_crd="out.crd",output_file="SP.out"):
        os.system("sander -O -i {0} -o {4} -p {1} -c {2} -r {3}".format(in_file,prmtop,inpcrd_file,output_crd,output_file))
        print "sander -O -i {0} -o {4} -p {1} -c {2} -r {3}".format(in_file,prmtop,inpcrd_file,output_crd,output_file)

        f = open(output_file)
        next = 0
        for line in f:
            if (next==1):
                next = 0
                energy = float(line.split()[1])
            if (re.search('NSTEP\s+ENERGY',line)):
                next = 1
        f.close()

        return energy

    def get_FDderv_AMBER(self,prmtop,cin_file,SPin_file,inpcrd_file,atoms,R_eq,RR):
        if (len(atoms)==2):
            small = 0.01
        elif (len(atoms)==3):
            small = 0.1
        elif (len(atoms)==4):
            small = 0.5
        val = self.get_internal(atoms) # evaluate the internal coordinate that will be frozen        

        ones = [1,-1]; energy = []
        for i in range(2):
            f = open("RST.dist",'w');        out = self.constraint_str(atoms,val+ones[i]*small);        f.write(out);        f.close()
            os.system("sander -O -i {0} -o output1 -p {1} -c {2} -r out1.rst".format(cin_file,prmtop,inpcrd_file))
            os.system("sander -O -i {0} -o output1_2 -p {1} -c out1.rst -r out2.rst".format(cin_file,prmtop))
            os.system("sander -O -i {0} -o output1_3 -p {1} -c out2.rst -r out3.rst".format(cin_file,prmtop))
            print "sander -O -i {0} -o output1 -p {1} -c {2} -r out1.rst".format(cin_file,prmtop,inpcrd_file)
            os.system("sander -O -i {0} -o out1 -p {1} -c out3.rst".format(SPin_file,prmtop))
            energy.append(self.get_AMBER_energy("out1"))

        derv = (energy[0] - energy[1])/(2*small)
        if ( (RR > R_eq and derv<0) or (RR < R_eq and derv>0)):
            print "problem with FD: ", RR,",",R_eq,energy[0],energy[1],derv
#            sys.exit(0)
        print "diff: ", energy[0], energy[1]

        return derv
    
    def get_AMBER_energy(self,file):
        f = open(file)
        next = 0
        for line in f:
            if (next==1):
                next = 0
                energy = float(line.split()[1])
            if (re.search('NSTEP\s+ENERGY',line)):
                next = 1
        f.close()

        return energy

    def append_pdb(self,prmtop,crd_file,pdb_file):
        os.system("ambpdb -p {0} < {1} >> {2} 2> /dev/null".format(prmtop,crd_file,pdb_file))

    def write_constraints(self,atoms,filename="RST.dist",extra=[],val=None,fixed=[],fixed_vals=[]):
        if (val==None):
            val = self.get_internal(atoms) # evaluate the internal coordinate that will be frozen

        # Write out the constraint file.
        f = open(filename,'w')
        if (len(extra)>0):
            for i,ex in enumerate(extra):
                out = self.constraint_str([ex-1,len(self.atoms)-len(extra)+i],0)
                f.write(out)

        for i in range(len(fixed)):
            out = self.constraint_str(fixed[i],fixed_vals[i])
            f.write(out)

        out = self.constraint_str(atoms,val)
        f.write(out)
        f.close()

    def write_all_constraints(self,atoms,filename="RST.dist",extra=[]):
        # Get the force constants and the increment. Different orders of magnatidue for bonds compared to angles and dihedals.

        # Write out the constraint file.
        f = open(filename,'w')
        if (len(extra)>0):
            for i,ex in enumerate(extra):
                out = self.constraint_str([ex-1,len(self.atoms)-len(extra)+i],0)
                f.write(out)

        out = ""
        for i in range(len(atoms)):
            val = self.get_internal(atoms[i][:-1]) # evaluate the internal coordinate that will be frozen
            out += self.constraint_str(atoms[i][:-1],val)
        f.write(out)
        f.close()

    def constraint_str(self,atoms,val):
        if (len(atoms)==2):
            small = 0.001
            fconst = 500000
        elif (len(atoms)==3):
            small = 0.01
            fconst = 2000000            
        else:
            small = 0.05
            fconst = 50000

        out = '&rst      iat = '
        out += ",".join(map(lambda x: str(x+1),atoms)) + ",\n"
        out += "r1 = {0: 6.4f}, r2 = {1: 6.4f}, r3 = {2: 6.4f}, r4 = {3: 6.4f},\n".format(val-small*2,val-small,val+small,val+small*2)
        out += "rk2 = {0}.0, rk3 = {0}.0,\n&end\n".format(fconst)

        return out

    def write_EPconstraints(self,atoms,filename="RST.dist"):
        # Get the force constants and the increment. Different orders of magnatidue for bonds compared to angles and dihedals.
        val = 0.0
        small = 0.0001
        fconst = 1000000

        # Write out the constraint file.
        f = open(filename,'a')
        f.write('&rst      iat = ')
        f.write(",".join(map(lambda x: str(x+1),atoms)) + ",\n")
        f.write("r1 = {0: 6.4f}, r2 = {1: 6.4f}, r3 = {2: 6.4f}, r4 = {3: 6.4f},\n".format(val-small*2,val-small,val+small,val+small*2))
        f.write("rk2 = {0}.0, rk3 = {0}.0,\n&end\n".format(fconst))
        f.close()


class amber_info:
    def __init__(self,name):
        self.name = name # prefix for file names
        self.natoms = 0
        self.parms = None
        self.xyz = []

    def read_inpcrd(self,filename):
        """ Read in an Amber .inpcrd file """
        f = open(filename,'r')
        f.readline()
        self.natoms = int(f.readline())
        xyz = []
        for line in f:
            L = line.split()
            xyz.extend(L)
        f.close()
        self.xyz = array(map(float,xyz))
        return self.xyz

    def write_inpcrd(self,filename):
        natoms = len(self.xyz)/3
        f = open(filename,'w')
        f.write("ZNC\n")
        f.write("{0:6d}\n".format(natoms))
        for i in range(natoms*3):
            f.write("{0: 12.7f}".format(float(self.xyz[i])))
            if (i % 6==5): f.write("\n")
        f.close()

    def atom_distance(self,i,j):
        x1 = self.xyz[3*i:3*(i+1)]
        x2 = self.xyz[3*j:3*(j+1)]

        return sqrt( dot(x1-x2,x1-x2) )

    def atom_angle(self,i,j,k):
        x1 = self.xyz[3*i:3*(i+1)]
        x2 = self.xyz[3*j:3*(j+1)]
        x3 = self.xyz[3*k:3*(k+1)]

        vec1 = x1 - x2
        vec2 = x3 - x2

        vec1 = vec1 / linalg.norm(vec1)
        vec2 = vec2 / linalg.norm(vec2)

        return math.acos(dot(vec1,vec2))

    def print_angle_terms(self):    
        Kb = map(float,self.parms['ANGLE_FORCE_CONSTANT'])
        EQb = map(float,self.parms['ANGLE_EQUIL_VALUE'])
        angles = map(int,self.parms['ANGLES_WITHOUT_HYDROGEN'])
        angle_info = []
        for i in range(len(angles)/4):
            at1 = angles[4*i]/3
            at2 = angles[4*i+1]/3
            at3 = angles[4*i+2]/3            
            angle = self.atom_angle(at1,at2,at3)
            index = angles[4*i+3]-1
            angle_info.append((index,at1,at2,at3,angle*180/math.pi,EQb[index]*180/math.pi,Kb[index]*(EQb[index]-angle)**2))

        st = sorted(angle_info, key=lambda student: student[6])
        for i in range(len(st),len(st)-10,-1):
            print st[i-1]

    def print_bond_terms(self):
        Kb = map(float,self.parms['BOND_FORCE_CONSTANT'])
        EQb = map(float,self.parms['BOND_EQUIL_VALUE'])
        bonds = map(int,self.parms['BONDS_WITHOUT_HYDROGEN'])
        bond_info = []
        for i in range(len(bonds)/3):
            at1 = bonds[3*i]/3
            at2 = bonds[3*i+1]/3
            dist = self.atom_distance(at1,at2)
            index = bonds[3*i+2]-1
            bond_info.append((index,at1,at2,dist,EQb[index],Kb[index]*(EQb[index]-dist)**2))

        st = sorted(bond_info, key=lambda student: student[5])
        for i in range(len(st),len(st)-10,-1):
            print st[i-1]
        
        
#            print at1," ",at2," ",Kb[index]*(EQb[index]-dist)**2
#            print dist,EQb[index]
#            print bond_info

    def read_parm_file(self):
        """ Read in an Amber .lib file """
        filename = self.name + ".prmtop"
        f = open(filename,'r')
        num = 0
        key = None
        data = {}
        for line in f:
            m = re.search('\%FLAG ([\w\_\d]+)',line)
            if (m):
                num = 1
                if (key):
                    data[key] = vals

                key = m.group(1)
                vals = []
            elif(num==1):
                num = num + 1
            elif(num==2):
                L = line.split()
                vals.extend(L)
        f.close()
        self.parms = data
        return data

class PDB:
    """ Contains all the properties of a molecule """
    def __init__(self,name):
        self.name = name
        self.atoms=[]
        self.residues=[]
        self.connect_array=[]

    def read_lib(self,filename):
        """ Read in an Amber .lib file """
        f = open(filename,'r')
        start=0
        natom=0
        resnum = 0
        nres = 0
        cgroup = ""
        for line in f:
            m = re.search('entry.([\w\d]+).unit.([\w\d]+)',line)
            if (m):
                if (not self.name):
                    self.name = m.group(1)
                cgroup = m.group(2)
                continue
            if (cgroup == 'residues'):
                L = line.split()
                self.residues[nres].name = L[0][1:-1]
                nres = nres + 1
            if (cgroup == 'connectivity'):
                L = line.split()
                self.connect_array.append(L[0:2])
            if (cgroup == 'atoms'):
                L = line.split()
                if (resnum!=L[3]):
                    self.residues.append(Residue("RES",L[3]))
                    resnum = L[3]

                m = re.match('\"([\w\d\*]+)\"',L[0]);                L[0] = m.group(1)
                m = re.match('\"([\w\d\*]+)\"',L[1]);                L[1] = m.group(1)                
                self.residues[-1].add_atom(rAtom(L[0],L[1],int(L[6]),[0,0,0],float(L[7])))

        f.close()

    def read_inpcrd(self,filename):
        """ Read in an Amber .inpcrd file """
        f = open(filename,'r')
        f.readline()
        f.readline()
        nres = 0
        natom = -1
        for line in f:
            L = line.split()
            count = 2 if (len(L)==6) else 1
            for i in range(count):
                if (len(self.residues[nres].atoms)==natom+1):
                    natom = 0
                    nres = nres + 1
                else:
                    natom = natom + 1
                self.residues[nres].atoms[natom].xyz = map(float,L[3*i:3*(i+1)])

    def print_pdb(self):
        for res in self.residues:
            res.print_res()

    def gen_gaussian_files(self,distance):
        """ Generates Gassuian files for determining charges on residues. All residue groups within distance are included."""

        # Find the close residues and clip out the QM part.
        # Remove bonded and angle MM atoms to prevent overpolarization.
        nres = len(self.residues)

        for i in range(nres):
            # Include the residue itself.
            qm_res = [0]*nres # 0 - MM, 1 - side chain QM, 10 - backbone, 20 - Nside, 30 - Cside only
            res = self.residues[i]
            qm_res[i] = 11
            # Include
            if (i > 0):
                qm_res[i-1] = 30
            if (i < len(self.residues)-1):
                qm_res[i+1] = 20
            self.close_groups(qm_res,i,distance) # Assign close groups.
            self.correct_qm_res(qm_res) # Make sure that the full residue is included if it's a GLY,ASP or PRO.
            self.correct_connections(qm_res) # Correct connections between groups.
            net_charge = self.net_charge(qm_res)
            qmmm = QMMM_atom_group()
            qmmm.add(self,qm_res)
#            qmmm.print_qm_system()
            qmmm.print_full_system(i)
            qmmm.write_gfile(net_charge,i)
#            qmmm.write_qin(i)

            # Only do RESP fitting on the current residue.
            (bcharge,scharge) = self.residues[i].net_charge()
            charge = bcharge+scharge
            qmmm.write_resp(i,charge)
            sys.exit(0)
#            if (i==16): sys.exit(0)

    def net_charge(self,qm_res):
        """ Returns the charge of the QM system """
        nres = len(qm_res)
        charge = 0
        for i in range(nres):
            (bcharge,scharge) = self.residues[i].net_charge()
            if (qm_res[i]>1): charge = charge + bcharge
            if (qm_res[i]%2==1): charge = charge + scharge
            
        return charge

    def correct_qm_res(self,qm_res):
        nres = len(qm_res)
        for i in range(nres):
            rname = self.residues[i].name[-4:-1]
            if (rname == "ASP" and qm_res[i]>0):   qm_res[i]=11
            if (rname == "GLY" and qm_res[i]==1):   qm_res[i]=11
            if (qm_res[i]>0 and len(self.residues[i].name)==4):   qm_res[i]=11

    def correct_connections(self,qm_res):
        nres = len(qm_res)
        for i in range(nres):
            if (qm_res[i]==10 or qm_res[i]==11): # backbone
                for j in range(i-1,0,-1):
                    if (qm_res[j]==20):
                        qm_res[j] = 10
                    elif (qm_res[j] < 10):
                        qm_res[j] = qm_res[j] + 20
                        break
                    elif (abs(qm_res[j]-30)<2): # correct structure
                        break
                for j in range(i+1,nres):
                    if (qm_res[j]==30):
                        qm_res[j] = 10
                    elif (qm_res[j] < 10):
                        qm_res[j] = qm_res[j] + 30
                        break
                    elif (abs(qm_res[j]-20)<2): # correct structure
                        break

    def close_groups(self,qm_res,res_num,distance):
        """ Find residues that are close to the residue 'res' """
        res = self.residues[res_num]

        for i in range(len(self.residues)):
            res1 = self.residues[i]
            for at in res1.atoms:
                if (qm_res[i]==11): break
                if (at.ignorable_atom(res.name)): continue
                backbone = at.is_backbone()
                if (backbone and (i==res_num-1 or i==res_num+1)): continue # don't include any more of the surround residue backbone atoms.
                if (backbone and qm_res[i] == 10): continue # skip if already including backbone.
                if (not backbone and qm_res[i] == 1): continue # skip if already including sidechain.
                dist = res.distance_closest(at,True)
#                if (res_num==16): print i,qm_res[i],dist,backbone,at.name
                if (dist > 7 + distance): break 
                if (dist < distance):
                    if (backbone):
                        qm_res[i] = 10 + (qm_res[i] % 2)
                    else:
                        if (qm_res[i] % 2 == 0): qm_res[i] = qm_res[i] + 1

class QMMM_atom_group:
    def __init__(self):
        self.mm_atype = []
        self.qm_elnum = []
        self.mm_xyz = []
        self.qm_xyz = []
        self.mm_charges = []
        self.qm_fixed = []
        self.qm_res = []
        self.qm_charges = []

    def add(self,pdb,qm_res):
        nres = len(qm_res)
        print qm_res
        for i in range(nres):
            # Add the full residue to the QM system.
            res= pdb.residues[i]
            if (qm_res[i]==11):
                for at in res.atoms:
                    self.add_QM_atom(at,cres=i)

            # Skip the backbone and only include the sidechain. The backbone is excluded from the MM system.
            elif (qm_res[i]==1):
                cb = res.get_atom("CB")
                for at in res.atoms:
                    if (at.name == "CA"):
                        self.add_QM_atom(at,cb,cres=i)
                    elif (not at.is_backbone()):
                        self.add_QM_atom(at,cres=i)

            # Only the backbone.
            elif (qm_res[i]==10):
                # Set the side chain as MM.
                self.add_sc_to_MM(res)
                
                # Link atom CA-CB -> CA-H.
                if (res.name != "GLY"):
                    cb = res.get_atom("CB")
                    self.add_QM_atom(at,cb,cres=i)

                # Add backbone to QM system.
                for at in res.atoms:
                    if (at.is_backbone()):
                        self.add_QM_atom(at,cres=1)
                        
            # The N side should be QM (20) and maybe the sidechain (21)
            elif (abs(qm_res[i]-20)<2):
                self.add_QM_cap(res,"NME",qm_res[i]-20==0) # need to fix the charges on NME
                
                if (qm_res[i]%2==0): # 20 - trim out close part of sidechain and add the SC as MM
                    self.add_sc_to_MM(res)
                else:
                    for at in res.atoms: # 21 - add the sidechain to the QM
                        if (not at.is_backbone()):  self.add_QM_atom(at)

            elif (abs(qm_res[i]-30)<2): # C side
                self.add_QM_cap(res,"ACE",qm_res[i]-30==0) # need to fix the charges on ACE

                if (qm_res[i]%2==0): # 30 - trim out close part of sidechain and add the SC as MM
                    self.add_sc_to_MM(res)
                else:
                    for at in res.atoms: # 31 - add the sidechain to the QM
                        if (not at.is_backbone()):  self.add_QM_atom(at)

            else: # 0 all MM
                for at in res.atoms:
#                    print "check",i,qm_res[i-1],at.name
                    if (i+1<nres and abs(qm_res[i+1] - 30)<2 and (at.name in ["C","O"] or res.name in [""])):
                        print "removing",res.name
                        continue
                    if (i>0 and abs(qm_res[i-1] - 20)<2 and (at.name in ["N","H"] or res.name in [""])):
                        continue
                    self.add_MM_atom(at)

    def add_QM_LA(self,res,name1,name2,fix=False,charge=0.0):
        """ Add atom CB as a hydrogen link atom. """
        ca = res.get_atom(name1)
        cb = res.get_atom(name2)
        if (cb==None): return
        self.add_QM_atom(cb,ca,fix,charge)

    def add_QM_cap(self,res,type,fullcap):
        """ Add QM NME or ACE cap. """
        if (type == "NME"):
            charge = { 'N': -0.4157, 'H': 0.2719, 'HA': 0.0976, 'CA': -0.1490 }
            if (fullcap): self.add_QM_LA(res,"CA","CB",True,charge['HA']) # Add CB as H.

            self.add_QM_LA(res,"CA","C",fullcap,charge['HA']) # add C from C=0 as H-CA.
            for at in res.atoms:
                if (at.name in ["N","H","CA","HA","HA2","HA3"]):
                    name = at.name[0:2] if (len(at.name)>2) else at.name
                    self.add_QM_atom(at,None,fullcap,charge[name])

        elif (type == "ACE"):
            charge = { 'C': 0.5972, 'O': -0.5679, 'HA': 0.1123, 'CA': -0.3662 }
                # Add CB as H.
            if (fullcap):    self.add_QM_LA(res,"CA","CB",True,charge['HA'])

            self.add_QM_LA(res,"CA","N",fullcap,charge['HA']) # add C from C=0 as H-CA.
            for at in res.atoms:
                if (at.name in ["C","O","CA","HA","HA2","HA3"]):
                    name = at.name[0:2] if (len(at.name)>2) else at.name
                    self.add_QM_atom(at,None,fullcap,charge[name])

    def add_sc_to_MM(self,res):
        if (res.name in ["SER","ASP","PRO","ILE"]):
            pass
        else:
            for at in res.atoms:
                if (at.name in ["HB2", "HB3", "CG", "CG1", "CG2","CB"]): continue # skip atoms close to backbone
                if (not at.is_backbone()): self.add_MM_atom(at)

    def add_QM_atom(self,at,at2=None,fix=False,ch=0.0,cres=-1):
        if (at2):
            self.qm_xyz.append(at.scale_LA(at2,1.09))
            self.qm_elnum.append(1)
        else:
            self.qm_xyz.append(at.xyz)
            self.qm_elnum.append(at.elnum)
        self.qm_res.append(cres)
        self.qm_fixed.append(fix)
        if (not fix): ch=0.0
        self.qm_charges.append(ch)

    def add_MM_atom(self,at):
        self.mm_xyz.append(at.xyz)
        self.mm_atype.append(at.element)
        self.mm_charges.append(at.charge)

    def qm_element(self,num):
        elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]
        
        return elements[self.qm_elnum[num]-1]

    def print_qm_system(self):
        f = open("qm_system.xyz",'w')
        f.write(str(len(self.qm_elnum)) + "\n\n")
        for i in range(len(self.qm_elnum)):
            xyz = map(float,self.qm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.qm_element(i),xyz[0],xyz[1],xyz[2]))
        f.close()

    def print_mm_system(self):
        f = open("mm_system.xyz",'w')
        f.write(str(len(self.mm_atype)) + "\n\n")
        for i in range(len(self.mm_atype)):
            xyz = map(float,self.mm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.mm_atype[i],xyz[0],xyz[1],xyz[2]))
        f.close()

    def print_full_system(self,num):
        f = open("full_system" + str(num) + ".xyz",'w')
        natoms = len(self.mm_atype)+ len(self.qm_elnum)
        f.write(str(natoms) + "\n\n")
        for i in range(len(self.qm_elnum)):
            xyz = map(float,self.qm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.qm_element(i),xyz[0],xyz[1],xyz[2]))
        for i in range(len(self.mm_atype)):
            xyz = map(float,self.mm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.mm_atype[i],xyz[0],xyz[1],xyz[2]))
        f.close()

    def write_gfile(self,net_charge,num):
        f = open("full" + str(num) + ".com",'w')
        f.write("%Nproc=8\n%NoSave\n#hf/6-31G* charge pop(CHELPG)\n\nTitle Card Required\n\n")
        f.write(str(net_charge) + " 1\n")

        # Write QM part
        for i in range(len(self.qm_elnum)):
            xyz = map(float,self.qm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.qm_element(i),xyz[0],xyz[1],xyz[2]))
        f.write("\n")

        # Write charges
        for i in range(len(self.mm_atype)):
            xyz = map(float,self.mm_xyz[i])
            f.write('{1:.4f} {2:.4f} {3:.4f} {0}\n'.format(self.mm_charges[i],xyz[0],xyz[1],xyz[2]))
        f.write("\n\n")
        f.close()

        f = open("resp" + str(num) + ".com",'w')
        f.write("%Nproc=8\n%NoSave\n#hf/6-31G* charge Pop=MK IOp(6/33=2,6/41=10,6/42=17)\n\nTitle Card Required\n\n")
        f.write(str(net_charge) + " 1\n")

        # Write QM part
        for i in range(len(self.qm_elnum)):
            xyz = map(float,self.qm_xyz[i])
            f.write('{0} {1:.4f} {2:.4f} {3:.4f}\n'.format(self.qm_element(i),xyz[0],xyz[1],xyz[2]))
        f.write("\n")
        # Write charges
        for i in range(len(self.mm_atype)):
            xyz = map(float,self.mm_xyz[i])
            f.write('{1:.4f} {2:.4f} {3:.4f} {0}\n'.format(self.mm_charges[i],xyz[0],xyz[1],xyz[2]))
        f.write("\n\n")
        f.close()

    def write_qin(self,num):
        f = open("qin" + str(num),'w')
        for i in range(len(self.qm_fixed)):
            if (self.qm_fixed[i]):
                f.write ("% 7.6f " % self.qm_charges[i])
            else:
                f.write ("% 7.6f " % 0.0)
            if (i>0 and i%8==0): f.write("\n")
        f.close()

    def write_resp(self,num,charge):

        # Write out the RESP file.
        f = open("resp" + str(num) + ".in",'w')
        f.write("floBF-resp run #1\n &cntrl,\n ihfree=1,\n qwt=0.0005,\n iqopt=1,\n /\n")
        f.write("    1.0\n")
        f.write("floB\n")
        natoms = 0
        for rnum in self.qm_res:
            if (rnum==num): natoms = natoms + 1
        print natoms
        f.write ('   {0:3d}  {1:3d}\n'.format(charge ,natoms))
        for i in range(len(self.qm_fixed)):
            if (self.qm_res[i]==num):
                f.write ("   %3d    0\n" % self.qm_elnum[i])            
#            if (self.qm_fixed[i]):
#                f.write ("   %3d   -1\n" % self.qm_elnum[i])
#            else:
#                f.write ("   %3d    0\n" % self.qm_elnum[i])
        f.write("\n")                
        f.close()

        # Write out the coordinates to take for RESP fitting from the full set.
        f = open("fixed_set" + str(num),'w')
        for rnum in self.qm_res:
            f.write(str(rnum==num) + "\n")
        f.close()

    def print_group():
        for i in range(len(self.charges)):
            print self.atype(i), " ", self.xyz(i), " ", self.charges(i)
        
        
class Residue:
    """ Defines all the characteristics an atom may have """
    def __init__(self,name,number,chain="A"):
        self.name = name # assign the basic types
        self.number = number
        self.chain = chain
        self.atoms = []

    def net_charge(self):
        sidecharge = 0
        if (self.name in ["ASP","GLU"]): sidecharge = -1
        if (self.name in ["ARG","LYS"]): sidecharge =  1

        if (len(self.name)==4 and self.name[0] == "N"):
            return 1,sidecharge
        elif (len(self.name)==4 and self.name[0] == "C"):
            return -1,sidecharge
        else:
            return 0,sidecharge

    def charge_on_residue(self):
        print self.name

    def add_atom(self,atom):
        self.atoms.append(atom)

    def get_atom(self,aname):
        for at in self.atoms:
            if (at.name == aname): return at

    def print_res(self):
        print self.name, " ", self.number, " ", self.chain
        for at in self.atoms:
            at.print_atom()

    def close_to_backbone(self,at0,dist):
        for at in self.atoms:
            if (at.is_backbone() and at.distance(at0)<dist):
                return True
        return False

    def distance_closest(self,at0,ignore=False):
        mindist = 100000.0
        for at in self.atoms:
            if (ignore and not at.ignorable_atom(self.name)): continue
            dist = at0.distance(at)
            if (dist<mindist):
                at1 = at
                mindist = dist

        return dist

class rAtom:
    """ Defines all the characteristics an atom may have """
    def __init__(self,name,atype,elnum,xyz,charge=0.0):
        self.name = name
        self.xyz = xyz
        self.atype = atype
        self.charge = charge
        self.elnum = elnum
        
        elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]
        self.element = elements[elnum-1]

    def distance(self,at):
        x1 = array(at.xyz)
        x2 = array(self.xyz)

        return sqrt( dot(x1-x2,x1-x2) )

    def ignorable_atom(self,rname):
        ignore = ["HG2", "HG3", "HG", "HB2", "CB", "HB3", "HB", "HA3", "HA2"]

        if (self.name in ignore):
            return True        
        elif (rname[-4:-1] == "PRO" and (self.name in ["CD","HD2","HD3"])):
            return True
        else:
            return False
        
    def print_atom(self):
        print 'Atom {0} {1:.4f} {2:.4f} {3:.4f} {4:.5f}'.format(self.element,self.xyz[0],self.xyz[1],self.xyz[2],self.charge)

    def scale_LA(self,at,dist):
        vec = array(self.xyz) - array(at.xyz)
        return array(at.xyz) + dist*vec/sqrt(dot(vec,vec))

    def is_backbone(self):
        backbone = ["CA", "N", "C", "O", "H", "HA", "HA2", "HA3"]
        return (self.name in backbone)

class Atom:
    """ Defines all the characteristics an atom may have """
    def __init__(self,aname,xyz,atype="",resid=1,resname="ZNC",charge=0.0,chain="A"):
        self.aname = aname # assign the basic types
        self.atype = atype
        self.xyz = xyz
        self.charge = charge
        self.resid = resid
        self.resname = resname
        self.chain = chain
        self.radius = 0.0
        self.elnum = -1
        
        matched = re.search( r'([a-zA-Z]{1,2})',aname )
        self.atom = matched.group(1)
        elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]

        if (re.search('EP',aname)):
            self.elnum = 50
            self.atom = "EP"
            return 

        for i in range(len(elements)):
            if (elements[i] == self.atom):
                self.atom = elements[i]
                self.elnum = i

        if (self.elnum == -1):
            for i in range(len(elements)):
                if (elements[i] == self.atom[0]):
                    self.atom = elements[i]
                    self.elnum = i
            if (self.elnum==-1):
#                print "Element ",self.atom," not found on periodic table."
                self.elnum=0
 #               sys.exit(0)


    def elnum_to_name(self,num):
        elements = ["H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Te","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe"]
        
        return elements[num]
        
    def mol2_str(self,num):
        return '{0:6d} {1:6s} {2:8.4f} {3:8.4f} {4:8.4f} {5:3s} {6:6d} {7:3s}   {8:8.5f}'.format(num+1,self.aname,self.xyz[0],self.xyz[1],self.xyz[2],self.atype,self.resid,self.resname,self.charge)

    def xyz_str(self):
        return '{0:3s} {1:9.5f} {2:9.5f} {3:9.5f}'.format(self.atom,self.xyz[0],self.xyz[1],self.xyz[2])

    def pdb_str(self,num):
#ATOM     28  C   ZNC     1     101.132  71.111  63.727
        if (len(self.aname)==4):
            return 'ATOM {0:6d} {1:4s}{2:4s}{7:2}{3:3d}     {4:7.3f} {5:7.3f} {6:7.3f}'.format(num+1,self.aname,self.resname,self.resid,self.xyz[0],self.xyz[1],self.xyz[2],self.chain)
        else:
            return 'ATOM {0:6d}  {1:3s} {2:4s}{7:2}{3:3d}     {4:7.3f} {5:7.3f} {6:7.3f}'.format(num+1,self.aname,self.resname,self.resid,self.xyz[0],self.xyz[1],self.xyz[2],self.chain)

    def print_atom(self):
        print 'Atom {0} {1:.4f} {2:.4f} {3:.4f} {4:.5f}'.format(self.atype,self.xyz[0],self.xyz[1],self.xyz[2],self.charge)

    def scale_bond_xyz(self,at,dist):
        xyz0 = array(map(float,at.xyz))
        xyz1 = array(map(float,self.xyz))
        vec = xyz1 - xyz0
        self.xyz = xyz0 + dist*vec/sqrt(dot(vec,vec))

    def distance_to_atom(self,at):
        xyz0 = array(map(float,at.xyz))
        xyz1 = array(map(float,self.xyz))
        vec = xyz1 - xyz0

        return sqrt(dot(vec,vec))

class gaussian_info(Molecule):
    def __init__(self,filename):
        """ Setup with g09 .log filename """
        Molecule.__init__(self,"g09com")
        self.g09filename = filename

    def read_esp_points(self,filename):
        f = open(filename,'r')
        f.readline()
        xyz = []; out = ""
        for line in f:
            L = line.split()
            xyz.append(L[-3:])
            out = out + "  ".join(L[-3:]) + "\n"
        f.close()

        return out

class gaussian_com():
    def __init__(self,filename):
        self.filename = filename
        self.xyz = []
        self.head = []
        self.tail = []
        self.atoms = []

    def read_head(self,f):
        for line in f:
            self.head.append(line)
            if re.match('[\-\d]+\s+[\-\d]+\s*$',line):
                L = line.split()
                self.charge = int(L[0])
                self.mult = int(L[1])
                return

    def get_internal(self,index):
        """ Get the internal coordinate value from the atoms 'index' numbers from 'self.xyz'. """
        mf = math_funs()
        if (len(index)==4):
            theta = mf.dihed_angle(self.xyz[index[0]],self.xyz[index[1]],self.xyz[index[2]],self.xyz[index[3]])
        elif (len(index)==3):
            theta = mf.bond_angle(self.xyz[index[0]],self.xyz[index[1]],self.xyz[index[2]])
        else:
            dR = array(self.xyz[index[0]]) - array(self.xyz[index[1]])
            theta = sqrt(dot(dR,dR))
            
        return theta

class gaussian_com_zmat(gaussian_com):
    def __init__(self,filename):
        gaussian_com.__init__(self,filename)
        self.connect = []
        self.zmat_vals = []
        
        f = open(self.filename,'r')
        self.read_head(f)
        self.read_zmat_data(f)
#        print self.atoms
 #       print self.connect
  #      print self.zmat_vals

    def read_zmat_data(self,f):
        """ Read in the z-matrix data from the .com file. """
        index = 0
        for line in f:
            index += 1
            L = line.split()
            if (len(L)<1): return
            self.atoms.append(L[0])
            self.connect.append([index-1])
            self.zmat_vals.append([])
            for i in range(3):
                if (len(L)>2*i+1):
                    self.zmat_vals[-1].append(L[2*i+2])
                    self.connect[-1].append(int(L[2*i+1])-1)

    def zmat_line(self,num,fixed=None):
        """ The Z-matrix line. """
        labels = ["B","A","D"]
        line = self.atoms[num] + "   "
        for i in range(1,len(self.connect[num])):
            scan_var = re.match('[A-Za-z]',self.zmat_vals[num][i-1])
                
            bond = self.get_internal(self.connect[num][0:(1+i)])
            fix = ( (fixed=="BONDS" and i==1) or (fixed=="ANGLES" and i<3))
            if (fix and not scan_var):
                line += "{0} {1}  ".format(self.connect[num][i]+1,bond)
            else:
                line += "{0} {1}{2}  ".format(self.connect[num][i]+1,labels[i-1],num)

        if (len(self.connect[num])==4): line += " 0 "

        return line

    def get_nvars(self):
        """ How many variables are there in z-matrix. """
        nvars = 0
        for vals in self.zmat_vals:
            for var in vals:
                if (re.match('[A-Za-z]',var)):
                    nvars +=1
        return nvars

    def get_scan_vars(self):
        """ How many variables are there in z-matrix. """
        vars = []; labels = ["B","A","D"]
        for i in range(len(self.zmat_vals)):
            for j in range(len(self.zmat_vals[i])):
                if (re.match('[A-Za-z]',self.zmat_vals[i][j])):
                    vars.append("{0}{1}".format(labels[j],i))
        return vars

    def zmat_variables(self,scan,fixed=None,stepsize=9,stepval=20.0):
        labels = ["B","A","D"]
        out = []; index = 0
        for i in range(len(self.zmat_vals)):
            for j in range(len(self.zmat_vals[i])):
                fix = ( (fixed=="BONDS" and j<1) or (fixed=="ANGLES" and j<2) )
                var = "{0}{1}".format(labels[j],i)
                val = self.get_internal(self.connect[i][0:(2+j)])
                mm = re.match('[A-Za-z]',self.zmat_vals[i][j])
                if (mm and scan==index):
                    out.append("{0}={1} S {2} {3:.1f}".format(var,val,stepsize,stepval))
                elif (not fix):
                    out.append("{0}={1}".format(var,val))
                if (mm): index +=1

        return out


    def write_fixed(self,updown,fixed=None,nsteps=9,stepsize=20,dir="./dihed"):
        """
        Set the bond values and assign variables to angles and dihedrals.
        Find the variables to scan and write the different files to be run forward and backward.
        """

        scan_vars = self.get_scan_vars()
        nscan = len(scan_vars)
        natoms = len(self.atoms)

        for scan in range(nscan):
            f = open("{1}{0}.com".format(scan,dir),'w')
            for hh in self.head:
                f.write(hh)
            for at in range(natoms):
                line = self.zmat_line(at,fixed)
                f.write(line + "\n")
            f.write("\n")
            tail_vars = self.zmat_variables(scan,fixed,stepval=updown[scan]*stepsize)
            for tt in tail_vars:
                f.write(tt + "\n")
            f.write("\n\n")                    
            f.close()
    

class gaussian_com_file(Molecule):
    def __init__(self,filename):
        """ Setup with g09 .log filename """
        Molecule.__init__(self,"g09com")
        self.g09filename = filename
        self.head = []
        self.tail = []
        self.charge = -99
        self.mult = -99        
        self.read_com(filename)

    def replace_head_str(self,org_str,new_str):
        for i in range(len(self.head)):
            self.head[i] = self.head[i].replace(org_str,new_str)

    def read_new_header(self,header_file):
        f = open(header_file,'r')
        self.head = f.readlines()
        f.close()

    def read_com(self,filename):
        """ Read in a Gaussian .com file """
        f = open(filename,'r')
        self.head = [];        self.tail = []; self.atoms = []
        xyz=0
        for line in f:
            L = line.split()
            if (xyz==0):
                self.head.append(line)
                if re.match('[\-\d]+\s+[\-\d]+\s*$',line):
                    xyz = 1
                    self.charge = int(L[0])
                    self.mult = int(L[1])
            if (xyz==1 and len(L)>3):
                self.atoms.append(Atom(L[0],map(float,L[1:4])))
            elif (xyz==1 and len(L)<2):
                xyz=2
            elif (xyz==2):
                if (len(line)>2): self.tail.append(line)

        f.close()

    def read_head(self,filename):
        f = open(filename,'r')
        self.head = []
        for line in f:
            self.head.append(line)
        f.close()

    def read_tail(self,filename):
        f = open(filename,'r')
        self.tail = []
        for line in f:
            self.tail.append(line)
        f.close()

    def write_com(self,filename):
        """ Writes out a .com file """
        f = open(filename,'w')

        # Write the head.
        for line in self.head:
            f.write(line)

        # Write the xyz coordinates.
        for index in range(len(self.atoms)):
            f.write(self.atoms[index].xyz_str() + "\n")

        # Write the tail.
        f.write("\n")
        for line in self.tail:
            f.write(line)
            
        f.write("\n\n")
        f.close()

class frcmod:

    def __init__(self,filename=None):
        self.filename = filename
        self.terms = {'bonds': [], 'angles': [], 'diheds': [], 'improps': [], 'mass': [], 'nonb': []}
        if (filename): self.read_frcmod()

    def copy_K(self,frc):
        head = ['BOND','ANGLE','DIHE']
        keys    = ['bonds','angles','diheds']
        for key in keys:
            for tt2 in frc.terms[key]:
                for i,tt in enumerate(self.terms[key]):
                    if (tt[0]==tt2[0]):
                        self.terms[key][i][1] = tt2[1]

    def get_terms(self,intern,mol,val='diheds'):
        # Get the terms that match up with the atom index numbers.
        st = [ mol.atoms[intern[0]].atype ]
        for i in range(1,len(intern)):
            st.append(mol.atoms[intern[i]].atype)
        st1 = '-'.join(st)
        st.reverse()
        st2 = '-'.join(st)
        out = []
        for term in self.terms[val]:
            if (st1==term[0] or st2==term[0]):
                out.append(term)
        return out

    def trim_to_subset(self,intern,mol):
        val = ['bonds','angles','diheds']
        new_terms = {'bonds': [], 'angles': [], 'diheds': [], 'improps': [], 'mass': [], 'nonb': []}        
        for int0 in intern:
            st = []
            for ii in int0:
                st.append(mol.atoms[ii].atype)
            st1 = '-'.join(st);            st.reverse();            st2 = '-'.join(st)
            for vv in val:
                for term in self.terms[vv]:
                    if (st1==term[0] or st2==term[0]):
                        new_terms[vv].append(term)

        self.terms = new_terms 

    def copy_zero_terms(self,frc):
        for bn in frc.terms['bonds']:
            if (float(bn[1])==0.0):
                self.terms['bonds'].append(bn)
        for an in frc.terms['angles']:
            if (float(an[1])==0.0):
                self.terms['angles'].append(an)
        for an in frc.terms['diheds']:
            if (float(an[2])==0.0):
                self.terms['diheds'].append(an)

    def set_K(self,K):
        nint = 0
        for bn in self.terms['bonds']:
            bn[1] = K[nint];            nint += 1
        for bn in self.terms['angles']:
            bn[1] = K[nint];            nint += 1
        for bn in self.terms['diheds']:
            bn[2] = K[nint];            nint += 1

    def augment_K(self,K):
        nint = 0
        for bn in self.terms['bonds']:
            bn[1] += K[nint];            nint += 1
        for bn in self.terms['angles']:
            bn[1] += K[nint];            nint += 1
        for bn in self.terms['diheds']:
            bn[2] += K[nint];            nint += 1

    def set_terms(self,eq_vals,nn):
        nint = 0

        for bn in self.terms['bonds']:
            bn[2] = eq_vals[nint];            nint += 1
        for an in self.terms['angles']:
            an[2] = eq_vals[nint];            nint += 1

        #        h4-cd-na-hn   1    1.700       180.000           2.000
        for an in self.terms['diheds']:
            an[3] = eq_vals[nint]
            an[4] = nn[nint]            
            nint += 1

    def calc_energy_terms(self,mol,internals):
        inc = [0,0,0]; out = []
        for ints in internals:
            int_type = len(ints[0])
            diff = 0
            cc = math.pi/180.0
            for int1 in ints:
                R = mol.get_internal(int1)
                if (int_type==2):
                    diff += (R - self.terms['bonds'][inc[int_type-2]][2])**2
                    print R, self.terms['bonds'][inc[int_type-2]][2]
                    print self.terms['bonds']
                    sys.exit(0)
                elif (int_type==3):
                    diff += (R*cc - self.terms['angles'][inc[int_type-2]][2]*cc)**2
#                    print R, self.terms['angles'][inc[int_type-2]][2],diff
                elif (int_type==4):
                    phase = self.terms['diheds'][inc[int_type-2]][3]
                    nn = self.terms['diheds'][inc[int_type-2]][4]
                    diff += (1 + math.cos(nn*R*cc - phase*cc))
            inc[int_type-2] += 1
            out.append(diff)

        return out
            

    def set_constants(self,eq_vals,k_bond,k_angle,k_dihe):
        nint = 0
        new_k = []
        for bn in self.terms['bonds']:
            if (float(bn[1])==0.0):
                bn[1] = "{0:6.2f}".format(k_bond)
            bn[2] = "{0:6.3f}".format(eq_vals[nint])
            nint += 1
#            if (nint==len(eq_vals)): return
        for an in self.terms['angles']:
            if (float(an[1])==0.0):
                an[1] = "{0:6.2f}".format(k_angle)
            an[2] = "{0:6.3f}".format(eq_vals[nint])
            nint += 1
#            if (nint==len(eq_vals)): return
        for an in self.terms['diheds']:
            if (float(an[2])==0.0 and float(an[3])==0.0):
                an[2] = "{0:6.3f}".format(k_dihe)
            an[3] = "{0:6.3f}".format(float(eq_vals[nint]))
            nint += 1
#            if (nint==len(eq_vals)): return            


    def read_frcmod(self):
        section = 0
        f = open(self.filename,'r')
        for ll in f:
            line = re.sub("^\s+","",ll)
            line = re.sub("([a-zA-z]+)\s+-","\\1-",line)
            line = re.sub("-\s+([a-zA-z]+)","-\\1",line)
            if re.match('MASS',line):
                section = 5
            elif re.match('NONBON',line):
                section = 6
            elif re.match('BOND',line):
                section = 1
            elif re.match('ANGLE',line):
                section = 2
            elif re.match('DIHE',line):
                section = 3
            elif re.match('IMPROPER',line):
                section = 4
            else:
                L = line.split()

                if (len(L)>1):
                    if (section==1):
                        ss = [ L[0],float(L[1]),float(L[2]) ]
                        self.terms['bonds'].append(ss)
                    elif (section==2):
                        ss = [ L[0],float(L[1]),float(L[2]) ]
                        self.terms['angles'].append(ss)
                    elif (section==3):
                        ss = [ L[0],int(L[1]),float(L[2]),float(L[3]),float(L[4]) ]
                        self.terms['diheds'].append(ss)
                    elif (section==4):
                        ss = [ L[0],float(L[1]),float(L[2]),float(L[3]) ]                        
                        self.terms['improps'].append(ss)
                    elif (section==5):
                        self.terms['mass'].append(L)
                    elif (section==6):
                        self.terms['nonb'].append(L[0:3])
        f.close()

    def read_subs(self,filename):
        f = open(filename,'r')
        subs = []
        for line in f:
            subs.append(line.split())
        f.close()

        return subs

    def duplicate_subs(self,subs):
#        newterms = {}
 #       for key in self.terms.keys():
  #          newterms[key] = []
   #         for val in self.terms[key]:
    #            newterms[key].append(val)

        for ss in subs:
            s1 = ss[0]
            for sval in ss[1:]:
                for key in self.terms.keys():
                    for line in self.terms[key]:
                        st = line[0]
                        count = st.count(sval)
                        if (count>0): # substituted term                        
                            cc = 0
                            for ss1 in ss:
                                if (ss1!=sval): cc += st.count(ss1)
                            if (cc==0): 
                                st1 = line[0].replace(sval,s1)
                                for key2 in self.terms.keys():
                                    for line2 in self.terms[key]:
                                        if (st1==line2[0]):
                                            line[1:] = line2[1:]

    def write(self,filename):
        f = open(filename,'w')
        head = ['MASS','BOND','ANGLE','DIHE','IMPROPER','NONBON']
        keys    = ['mass','bonds','angles','diheds','improps','nonb']
        f.write("\nMASS\n")
        for kk in self.terms['mass']:
            if (len(kk)==2):
                f.write("{0} {1:10.3f}\n".format(kk[0],float(kk[1])))
            else:
                f.write("{0} {1:6.3f} {2:10.3}\n".format(kk[0],float(kk[1]),float(kk[2])))
        f.write("\nBOND\n")
        for kk in self.terms['bonds']:
            f.write("{0:7s} {1:8.3f} {2:8.3f}\n".format(kk[0],float(kk[1]),float(kk[2])))
        f.write("\nANGLE\n")
        for kk in self.terms['angles']:
            f.write("{0:11s} {1:8.3f} {2:8.3f}\n".format(kk[0],float(kk[1]),float(kk[2])))
        f.write("\nDIHE\n")
        for kk in self.terms['diheds']:
            f.write("{0:15s} {1:5d} {2:8.3f} {3:8.3f} {4:8.3f}\n".format(kk[0],int(float(kk[1])),float(kk[2]),float(kk[3]),float(kk[4])))
        f.write("\nIMPROPER\n")
        for kk in self.terms['improps']:
            f.write("{0:15s} {1:8.3f} {2:8.3f} {3:8.3f}\n".format(kk[0],float(kk[1]),float(kk[2]),float(kk[3])))
        f.write("\nNONBON\n")
        for kk in self.terms['nonb']:
            f.write("{0:7s} {1:8.3f} {2:8.3f}\n".format(kk[0],float(kk[1]),float(kk[2])))
            
#        for i,key in enumerate(keys):
#
#            for line in self.terms[key]:
#                f.write(" ".join(line) + "\n")
        f.write("\n\n")
        f.close()

    def get_average_internals(self,mol,internals):
        """ Get the average values for the internal coordinate. Also get the phase factor for the dihedrals. """
        # eq_vals
        eq_vals = []
        
        # Loop through the internals.
        for i,ints in enumerate(internals):
            int_type = len(ints[0])
            vals = []
            for int1 in ints:
                vals.append(mol.get_internal(int1))
            val = average(vals)
            st = std(vals)
            n2=0;n3=0
            if (int_type==4):
                nn =  self.terms['diheds'][i-len(self.terms['bonds'])-len(self.terms['angles'])][4]
                eq_vals.append((nn*vals[0]-180) % 360)
#                print "nn: ",nn
            else:
                eq_vals.append(vals[0])

#            print vals
#            print ints
#            print eq_vals[-1]

        return eq_vals

    def get_first_internals(self,mol,internals):
        """ Get the average values for the internal coordinate. Also get the phase factor for the dihedrals. """
        eq_vals = []
        
        # Loop through the internals.
        for i,ints in enumerate(internals):
            int_type = len(ints[0])
            val = mol.get_internal(ints[0])
            print "val : ", val, ints[0]
            if (int_type==4):
                nn =  self.terms['diheds'][i-len(self.terms['bonds'])-len(self.terms['angles'])][4]
                eq_vals.append((nn*val-180) % 360)
            else:
                eq_vals.append(val)

        return eq_vals


    def move_list(self,bonds,angles,diheds,mol):
        """ Setup a perturb list. """

        # Bounds to be moved.
        bonds_move = []
        for bn in self.terms['bonds']:
            ats = bn[0].split('-')
            bonds_move.append([])
            for bond in bonds:
                b1 = mol.atoms[bond[0]].atype
                b2 = mol.atoms[bond[1]].atype                
                if ((b1 == ats[0] and b2 == ats[1]) or (b2 == ats[0] and b1 == ats[1])):
                    bonds_move[-1].append(bond)
            if (len(bonds_move[-1])<1):
                print "Could not find bond: ", bn[0]

        # Angles to be moved.
        angles_move = []
        for an in self.terms['angles']:
            ats = an[0].split('-')
            angles_move.append([])
            for angle in angles:
                a1 = mol.atoms[angle[0]].atype
                a2 = mol.atoms[angle[1]].atype
                a3 = mol.atoms[angle[2]].atype
                if ((a1 == ats[0] and a2 == ats[1] and a3 == ats[2]) or (a3 == ats[0] and a2 == ats[1] and a1 == ats[2])):
                    angles_move[-1].append(angle)
            if (len(angles_move[-1])<1):
                print "Could not find angle: ", an[0]

        # Dihedrals to be moved.
        diheds_move = []
        skip = []
        nn_dihed = []
        dihe = self.terms['diheds']

        # Trim out duplicates.
        for i in range(len(dihe)):
            found = False
            for j in range(i):
                if (dihe[i][0]==dihe[j][0]):
                    found = True
            skip.append(found)
        
        for i,an in enumerate(dihe):
            if (skip[i]): continue
            L = an[0].split('-')
            skip.append(False)
            nn_dihed.append(max(int(an[4]),1))
            for j in range(i+1,len(dihe)):
                an2 = dihe[j]
                L2 = an2[0].split('-')
                if ((L[1] == L2[1] and L[2] == L2[2]) or (L[1] == L2[2] and L[2] == L2[1])):
                    skip[-1] = True
            ats = an[0].split('-')
            diheds_move.append([])
            for dihed in diheds:
                a1 = mol.atoms[dihed[0]].atype
                a2 = mol.atoms[dihed[1]].atype
                a3 = mol.atoms[dihed[2]].atype
                a4 = mol.atoms[dihed[3]].atype
                if ((a1 == ats[0] and a2 == ats[1] and a3 == ats[2] and a4 == ats[3])
                    or (a4 == ats[0] and a3 == ats[1] and a2 == ats[2] and a1 == ats[3])):
                    diheds_move[-1].append(dihed)
            if (len(diheds_move[-1])<1):
                print "Could not find dihedral: ", an[0]
                
        return bonds_move,angles_move,diheds_move,skip,nn_dihed

class gaussian_oniom_com(gaussian_info):
    def __init__(self,filename):
        """ Setup with g09 .log filename """
        gaussian_info.__init__(self,"ONIOM")

        # New terms for a Gaussian_oniom com file.
        self.LH = []; self.link = []
        self.head_info = [];        self.tail = []
        self.ONfilename = filename
        self.real_low_charge = ""
        self.high_charge = ""
        self.model_low_charge = ""

    def __str__(self):
        out = ""
        for i in range(len(self.atoms)):
            at = self.atoms[i]
            out = out + "{0}-{1}-{2:.6f}        {3: .6f} {4: .6f} {5: .6f}\n".format(at.aname,at.atype,float(at.charge),float(at.xyz[0]),float(at.xyz[1]),float(at.xyz[2]))

        for i in range(len(self.bond_list)):
            out = out + "{0} {1} {2}\n".format(self.bond_type[i],self.bond_list[i][0],self.bond_list[i][1])

        return out

    def get_elec_interact(self):
        natoms = len(self.LH)
        sum = 0
        for i in range(natoms):
            print self.LH[i]
            if (self.LH[i]=="L"):
                for j in range(natoms):
                    if (self.LH[j]=="H"):
#                        print i,j,self.atoms[i].xyz,self.atoms[j].xyz
                        rij = self.rij(j,i)
                        sum += self.atoms[i].charge*self.atoms[j].charge/rij

        return sum

    def next_atom(self,at1,at2):
        bonds = self.bond_matrix[at2]
        out = []
        for bb in bonds:
            if (bb!=at1 and bb!=at2 and self.atoms[bb].elnum>1):
                out.append(bb)

        return out

    def get_link(self,t1,t2):
        """ Assume the first atom is a hydrogen and work backward to link atom. """
        natoms = len(self.atoms)
        con = [ natoms-1 ]
        newcon = [ natoms -1 ]
        for i in range(natoms-1):
            new1 = copy(newcon)
            newcon = []
            for c1 in new1:
                for cc in self.bond_matrix[c1]:
                    if (not cc in con and self.atoms[cc].elnum>1):
                        newcon.append(cc)
                        con.append(cc)
                        if (self.atoms[cc].atype==t1 and self.atoms[c1].atype==t2):
                            return [ cc, c1 ]

        return [ -1, -1 ]
        

    def build_oniom_file(self,eq_charge=False,tail=None):
        header = "#P ONIOM(hf/genECP:AMBER=hardfirst)=EmbedCharge geom=(connectivity) nosymm iop(2/15=3) pop=chelpg opt(modred)\n\ncomment\n\n"
#        header = "#P ONIOM(hf/gen:AMBER=hardfirst)=(EmbedCharge,scalecharge=555) geom=(connectivity) nosymm iop(2/15=3) opt(QuadMac,readfreeze)\n\ncomment\n\n";
        high = []; check = []; cut = []; cut2 = []
        for i in range(1000):
            high.append(0)
        
        f = open("link.atoms",'r')
        for line in f:
            L = line.split()
            high[int(L[1])-1] = 1
            cut.append(int(L[0]))
            cut2.append(int(L[1]))            
            if (not L[1] in check):
                check.append(L[1])

        # Get the array 'high'.
        print cut
        while(len(check)>0):
            atom = int(check.pop())
            for bl in self.bond_list:
                if (bl[0]==atom and high[bl[1]-1]==0 and (not bl[1] in cut)):
                    high[bl[1]-1] = 1
                    check.append(bl[1])
                    print "adding1:", bl[1]
                elif (bl[1]==atom and high[bl[0]-1]==0 and (not bl[0] in cut)):
                    high[bl[0]-1] = 1
                    check.append(bl[0])
                    print "adding2:", bl[0]

        # Figure out the charge of the system.
        high_charge = 0;     low_charge = 0        ; nlow =0
        for i in range(len(self.atoms)):
            at = self.atoms[i]
            if (high[i]):
                high_charge = high_charge + at.charge
            else:
                low_charge = low_charge + at.charge
                nlow = nlow + 1

        # Correction to the total charge of the the high system to give an integer.
        print "High Charge: ", high_charge
        int_hc = int(round(high_charge))
        int_lc = int(round(low_charge ))
        hcharge = int_hc - high_charge
        lcharge = int_lc -  low_charge

        # Equilibrate the MM charge
        if (eq_charge and nlow > 0):
            for i in range(len(self.atoms)):
                if (not high[i]):
                    self.atoms[i].charge = self.atoms[i].charge + float(lcharge/nlow)
            
        # Write out the ONIOM file.
        f = open(self.ONfilename,'w')
        f.write(header)
        f.write("{0:d} 1 {1:d} 1 {1:d} 1\n".format(int_hc+int_lc,int_hc))
        for i in range(len(self.atoms)):
            at = self.atoms[i]
            if (i+1 in cut):
                j = cut.index(i+1)
                f.write(' {0}-{1}-{2:-5.5f}    0     {3: 9.6f} {4: 9.6f} {5: 9.6f}    L H-HC-{6:-6.5f} {7}\n'.format(at.atom,at.atype,at.charge,at.xyz[0],at.xyz[1],at.xyz[2],hcharge/len(cut),cut2[j]))
            else:
                if (high[i]==1):
                    f.write(' {0}-{1}-{2:-5.5f}    0     {3: 9.6f} {4: 9.6f} {5: 9.6f}       H\n'.format(at.atom,at.atype,at.charge,at.xyz[0],at.xyz[1],at.xyz[2]))
                else:
                    f.write(' {0}-{1}-{2:-5.5f}    0     {3: 9.6f} {4: 9.6f} {5: 9.6f}       L\n'.format(at.atom,at.atype,at.charge,at.xyz[0],at.xyz[1],at.xyz[2]))
        f.write('\n')
        for i in range(len(self.atoms)):
            full = []
            for bl in self.bond_list:
                if (bl[0]==i+1 and bl[1]>i+1): full.append(bl[1])
                if (bl[1]==i+1 and bl[0]>i+1): full.append(bl[0])
            f.write(" " + str(i+1) + " ")
            for j in full:
                f.write("{0} 1.0 ".format(j))
            f.write('\n')
        f.write('\n')
        if (tail!=None):
            for tt in tail:
                f.write(tt)
        f.close()


    def link_atom_position(self):
        # Find position of link atom.
        for i in range(len(self.atoms)):
            if (self.link[i]):
                LAindex = i
                break

        # Find the closest atom in the H part.
        for i in range(len(self.atoms)):
            dist = self.atom_distance(LAindex,i)

    def set_radii(self):
        f = open("/home/skb10/python/modules/radii.txt",'r')
        periodic = []
        for line in f:
            periodic.append(line.split(','))
            
        for at in self.atoms:
            at.radius = float(periodic[at.elnum][3])

        f.close()

    def trim_esp_close_to_MM(self,esp,mult,radius):
        espout = []; espxyz = []; xyz = []
        espxyz0 = esp.split('\n')
        radii = []

        self.set_radii()

        for i in range(len(espxyz0)):
            lst = re.split('\s+',espxyz0[i])
            if (len(lst)>1):
                espxyz.append(map(float,lst))

        at = []
        for i in range(len(self.atoms)):
            if (self.LH[i] == "L"):
                xyz.append(map(float,self.atoms[i].xyz))
                radii.append(self.atoms[i].radius)
                at.append(self.atoms[i].atom)

        out = ""; out2 = ""; nout = 0; out3 = ""; nout2 = 0
        for i in range(len(espxyz)):
            skip = False
            for j in range(len(xyz)):
                vec = array(espxyz[i]) - array(xyz[j])
                dist = sqrt(dot(vec,vec))
                if (dist<mult*radii[j]):
                    skip = True
                    break
            if (not skip):
                out = out + "  ".join(map(str,espxyz[i])) + "\n"
                nout = nout + 1
                out2 = out2 + "    1      1   {0:.7f}  {1:.7f}  {2:.7f}\n".format(espxyz[i][0],espxyz[i][1],espxyz[i][2])
            else:
                nout2 = nout2 + 1
                out3 = out3 + "    1      1   {0:.7f}  {1:.7f}  {2:.7f}\n".format(espxyz[i][0],espxyz[i][1],espxyz[i][2])

        f = open("main_ESP.pts",'w')
        f.write("      {0}\n".format(nout))
        f.write(out2)
        f.close()
        f = open("trimmed_ESP.pts",'w')
        f.write("      {0}\n".format(nout2))
        f.write(out3)
        f.close()
        
        return out

    def linked_atom(self,link_index):
        """ Find the atom that the link atom is linked with. """
        ind = -1
        for i in range(len(self.bond_list)):
            bl = self.bond_list[i]
            if (link_index == bl[0]-1):
                ind = bl[1] - 1
                if (self.LH[ind]=="H"):   break
            if (link_index == bl[1]-1):
                ind = bl[0] - 1
                if (self.LH[ind]=="H"):   break

        return ind

    def link_xyz(self,link_index):
        """ Linked atom xyz coordinates """
        la = self.linked_atom(link_index)
        xyz0 = array(map(float,self.atoms[la].xyz))
        xyz1 = array(map(float,self.atoms[link_index].xyz))
        vec = xyz1 - xyz0
        return xyz0 + 1.08*vec/sqrt(dot(vec,vec))

    def write_com(self,xyz,filename,move_xyz=False):
        """ Write out an ONIOM com file. """
        f = open(filename,'w')
        # Write the head info
        for head in self.head_info:
            f.write(head)

        # Write the body. The xyz coordinates.
        for i,at in enumerate(self.atoms):
            if (self.LH[i]=="H" and move_xyz):
                xyz[i][1] += 300
            link = self.link[i]
            if (link==None):
                f.write("{0}-{1}-{2}  {3: 7.5f}  {4: 7.5f}  {5: 7.5f}   {6}\n".format(at.aname,at.atype,at.charge,xyz[i][1],xyz[i][2],xyz[i][3],self.LH[i]))
            else:
                f.write("{0}-{1}-{2}  {3: 7.5f}  {4: 7.5f}  {5: 7.5f}   {6} {7}-{8}-{9} {10}\n".format(at.aname,at.atype,at.charge,xyz[i][1],xyz[i][2],xyz[i][3],self.LH[i],link[0],link[1],link[2],link[3]))

        # Write the connectivity information.
        f.write("\n")
        for con in self.connect_info:
            f.write(con)
        f.write("\n")
        # Write the tail.
        for tail in self.tail:
            f.write(tail)
        f.close()
        
    def read_in_ONIOM_com_info(self):
        section = 0
        self.connect_info = []

        f = open(self.ONfilename)
        for line in f:
            if (section==0): # header info
                self.head_info.append(line)
                m = re.search('^\s*([\d\-]+\s+[\d\-]+)\s+([\d\-]+\s+[\d\-]+)\s+([\d\-]+\s+[\d\-]+)',line)
                if (m):
                    self.real_low_charge = m.group(1)
                    self.high_charge = m.group(2)
                    self.model_low_charge = m.group(3)
                    section = 1
            elif (section==1): # atoms
                L = line.split()
                if (len(L)<3):
                    section = 2
                else:
                    m = re.search('^\s*(\w+)\-([\w\*\d]+)\-([\-\d\.]+)\s+\d*\s+([\-\d\.]+)\s+([\-\d\.]+)\s+([\-\d\.]+)\s+(\w+)\s*(\S*\s*\d*)',line)
                    aname = m.group(1)
                    atype = m.group(2)
                    charge = float(m.group(3))
                    xyz = map(float,[m.group(4),m.group(5),m.group(6)])
                    self.atoms.append(Atom(aname,xyz,atype,0,"0",charge))
#                    print "-> ", aname, atype, charge,self.atoms[-1].charge
                    self.LH.append(m.group(7))
#                    print "yo: ", m.group(7),m.group(8)
                    if (m.group(8)):
                        m2 = re.search('(\S+)\-([\w\d]+)\-([\d\-\.]+)\s+(\d+)',m.group(8))
                        self.link.append([m2.group(1),m2.group(2),m2.group(3),m2.group(4)])
                    else:
                        self.link.append(None)
            elif(section==2): # connection matrix
                tline = re.sub('^\s+','',line)
                L = tline.split()
                if (len(L)<1):
                    section = 3
                else:
                    self.connect_info.append(line)
                    at = L.pop(0)
                    for i in range(len(L)):
                        if (i%2==0):
                            self.bond_list.append(map(int,[at,L[i]]))
                        else:
                            self.bond_type.append(L[i])
            elif(section==3): # tail info
                self.tail.append(line)
        f.close()

    def replace_xyz(self,xyz):
        natoms = len(self.atoms)

        for i in range(natoms):
            self.atoms[i].xyz = xyz[i]

    def build_charge_file(self,filename):
        # The Header.
        f = open(filename,'w')
        f.write("#P HF/6-31++G** Density=SCF Charge NoSymm\n\n comment\nCharge file\n\n")
        line = re.sub('^\s+','',self.head_info[-1])
        L = line.split()
        f.write('{0} {1}\n'.format(L[0],L[1]))

        #Build the a new .com file with the xyz coordinates replaced.
        charges = []
        for i in range(len(self.atoms)):
            at = self.atoms[i]
            xyz = at.xyz
            if (self.link[i]):
                xyz = self.link_xyz(i)
                f.write("{0}        {1: .6f} {2: .6f} {3: .6f}\n".format("H",float(xyz[0]),float(xyz[1]),float(xyz[2])))                
            elif (self.LH[i]=="H"):
                f.write("{0}        {1: .6f} {2: .6f} {3: .6f}\n".format(at.aname,float(xyz[0]),float(xyz[1]),float(xyz[2])))
            else:
                charges.append("{0: .6f} {1: .6f} {2: .6f}   {3}\n".format(float(xyz[0]),float(xyz[1]),float(xyz[2]),at.charge))
        f.write("\n")
        for cc in charges:
            f.write(cc)
        f.write("\n\n")            

    def build_real_MM(self,filename):
        # The Header.
        f = open(filename,'w')
        f.write("#P Test IOp(2/15=1,5/32=2,5/38=1) AMBER\n Geom=Connect\n\n comment\n Point  3 -- low level on real system.\n\n")
        line = re.sub('^\s+','',self.head_info[-1])
        L = line.split()
        f.write('{0} {1}\n'.format(L[0],L[1]))

        #Build the a new .com file with the xyz coordinates replaced.
        for i in range(len(self.atoms)):
            at = self.atoms[i]
            charge = 0.0 if (self.LH[i]=="H") else float(at.charge)
            xyz = at.xyz
            f.write("{0}-{1}-{2:.6f}        {3: .6f} {4: .6f} {5: .6f}\n".format(at.aname,at.atype,charge,float(xyz[0]),float(xyz[1]),float(xyz[2])))

        # Print out the tail.
        f.write('\n')
        b_info = self.format_oniom_bond_info()
        f.write(b_info)
        f.write('\n\n')
        f.close()

    def format_oniom_bond_info(self):
        out = ""
        for i in range(len(self.atoms)):
            out = out + " {0:d}".format(i+1)
            for j in range(len(self.bond_list)):
                if (self.bond_list[j][0] == i+1):
                    out = out + ' {0:d} {1}'.format(self.bond_list[j][1],self.bond_type[j])
            out = out + "\n"

        return out

    def trim_to_QM(self):
        natoms = len(self.atoms)
        index = []; nQM = 0
        rev_index = []
        for i in range(natoms):
            rev_index.append(-1)
        gCOM = gaussian_oniom_com(self.ONfilename)
        
        for i in range(natoms):
            if (self.LH[i] == "H"):
                index.append(i)
                rev_index[i] = nQM
                gCOM.atoms.append(self.atoms[i])
                nQM = nQM + 1
        for i in range(len(self.bond_list)):
            for ii in range(nQM):
                [b1,b2] =self.bond_list[i]
                if (b1 == index[ii]+1 and rev_index[b2-1]>-1): # link atom not included in bonding pattern.
                    gCOM.bond_type.append(self.bond_type[i])
                    gCOM.bond_list.append([ii+1,rev_index[b2-1]+1])

        return gCOM

    def trim_to_link_QM(self):
        natoms = len(self.atoms)
        gau = gaussian_info(Molecule)
        index = []; nQM = 0; link_index = -1
        rev_index = []
        for i in range(natoms):
            rev_index.append(-1)

        # Find the link atom.
        for i in range(natoms):
            if (self.link[i]):
                link_index = i

        # Linked atoms.
        extra = [False]*natoms
        for i in range(len(self.bond_list)):
            [b1,b2] =self.bond_list[i]
#            print [b1,b2]
            if (int(b1) == link_index + 1):
                extra[b2-1] = True
            if (int(b2) == link_index + 1):
                extra[b1-1] = True

        # Keep QM system plus 1 of connected atoms.
        for i in range(natoms):
            if (self.LH[i] == "H" or extra[i] or self.link[i]):
                gau.atoms.append(self.atoms[i])
                if (self.LH[i] == "L" and extra[i]):
                    gau.atoms[-1].scale_bond_xyz(self.atoms[link_index],1.08)
                    gau.atoms[-1].atom = "H"
                    gau.atoms[-1].aname = "H"

        return gau

    def trim_to_QMMM(self):
        natoms = len(self.atoms)
        gau = gaussian_info(Molecule)
        index = []; nQM = 0; link_index = -1
        rev_index = []
        for i in range(natoms):
            rev_index.append(-1)

        # Find the link atom.
        for i in range(natoms):
            if (self.link[i]):
                link_index = i

        # Keep QM system plus 1 of connected atoms.
        for i in range(natoms):
            if (self.LH[i] == "H" or self.link[i]):
                gau.atoms.append(self.atoms[i])
                if (self.link[i]):
                    gau.atoms[-1].scale_bond_xyz(self.atoms[link_index],1.08)
                    gau.atoms[-1].atom = "H"
                    gau.atoms[-1].aname = "H"

        return gau

    def trim_to_link_MM(self):
        natoms = len(self.atoms)
        gau = gaussian_info(Molecule)
        index = []; nQM = 0; link_index = -1
        rev_index = []
        for i in range(natoms):
            rev_index.append(-1)

        # Find the link atom.
        for i in range(natoms):
            if (self.link[i]):
                link_index = i

        link = int(self.link[link_index][3])-1
        
        # Keep QM system plus 1 of connected atoms.
        for i in range(natoms):
            if (self.LH[i] == "L"):
                gau.atoms.append(self.atoms[i])
            elif (i == link):
                gau.atoms.append(self.atoms[i])
                gau.atoms[-1].scale_bond_xyz(self.atoms[link_index],1.08)
                gau.atoms[-1].atom = "H"
                gau.atoms[-1].aname = "H"

        return gau

    def trim_to_methyl(self):
        natoms = len(self.atoms)
        gau = gaussian_info(Molecule)
        index = []; nQM = 0; rev_index = [-1]*natoms; link_index = -1

        # Find the link atom.
        for i in range(natoms):
            if (self.link[i]):
                link_index = i
                gau.atoms.append(self.atoms[i])
                break

        # Keep QM system plus 1 of connected atoms.
        for i in range(len(self.bond_list)):
            [b1,b2] = self.bond_list[i]
            if (b1-1 == link_index):
                gau.atoms.append(self.atoms[b2-1])
                gau.atoms[-1].scale_bond_xyz(self.atoms[link_index],1.08)
                gau.atoms[-1].atom = "H"
                gau.atoms[-1].aname = "H"
            if (b2-1 == link_index):
                gau.atoms.append(self.atoms[b1-1])
                gau.atoms[-1].scale_bond_xyz(self.atoms[link_index],1.08)
                gau.atoms[-1].atom = "H"
                gau.atoms[-1].aname = "H"

        return gau

    def trim_to_link_QMMM(self):
        natoms = len(self.atoms)
        gau = gaussian_info(Molecule)
        index = []; nQM = 0; rev_index = [-1]*natoms; link_index = -1

        # Find the link atom.
        for i in range(natoms):
            if (self.link[i]):
                link_index = i

        # Keep QM system plus 1 of connected atoms.
        con = []; skip = []
        for i in range(len(self.bond_list)):
            [b1,b2] =self.bond_list[i]
            if (b1 == link_index+1):
                if (self.LH[b2-1]=="L"):
                    if (self.atoms[b2-1].atom != "H"):
                        con.append(b2-1)
                    else:
                        skip.append(b2-1)
                
            if (b2 == link_index+1):
                if (self.LH[b1-1]=="L"):
                    if (self.atoms[b1-1].atom != "H"):
                        con.append(b1-1)
                    else:
                        skip.append(b1-1)

        for i in range(natoms):
            if (i == link_index):
                pass
            elif (self.LH[i] == "L"):
                if (i in skip): continue
                gau.atoms.append(self.atoms[i])
                if (i in con):
                    at = Atom("H",map(float,self.atoms[link_index].xyz))
                    at.scale_bond_xyz(self.atoms[i],1.01)
                    gau.atoms.append(at)

        return gau

    def build_model_MM(self,filename):
        # The Header.
        trimmed_mol = self.trim_to_QM()
        f = open(filename,'w')
        f.write("#P Test IOp(2/15=1,5/32=2,5/38=1) AMBER\n Geom=Connect\n\n comment\n Point  3 -- low level on real system.\n\n")
        line = re.sub('^\s+','',self.head_info[-1])
        L = line.split()
        f.write('{0} {1}\n'.format(L[2],L[3]))

        #Build the a new .com file with the xyz coordinates replaced.
        for i in range(len(trimmed_mol.atoms)):
            at = trimmed_mol.atoms[i]
            xyz = at.xyz
            f.write("{0}-{1}-{2:.6f}        {3: .6f} {4: .6f} {5: .6f}\n".format(at.aname,at.atype,0.0,float(xyz[0]),float(xyz[1]),float(xyz[2])))
        f.write("{0}-{1}-{2:.6f}        {3: .6f} {4: .6f} {5: .6f}\n".format("H","HC",0.0,100.0,100.0,100.0))

        # Print out the tail.
        f.write('\n')
        b_info = trimmed_mol.format_oniom_bond_info()
        f.write(b_info)
        f.write(" {0}\n".format(1+len(trimmed_mol.atoms)))
        f.write('\n\n')

    def build_scf_of_model(self,filename):
        # The Header.
        f = open(filename,'w')
        f.write("#P hf/sto-2g NOSYMM scrf(PCM,Read)\n\n comment\n\n")
        f.write(self.high_charge + "\n")

        #Build the a new .com file with the xyz coordinates replaced.
        for i in range(len(self.atoms)):
            if (self.LH[i] == "H" or self.link[i]):
                at = self.atoms[i]
                if (self.link[i]):
                    xyz = self.link_xyz(i)
                    xyz = at.xyz
                    name = "H"
                else:
                    xyz = at.xyz
                    name = at.aname
                f.write("{0}       {1: .6f} {2: .6f} {3: .6f}\n".format(name,float(xyz[0]),float(xyz[1]),float(xyz[2])))

        # Print out the tail.
        f.write('\nGeomView\nPDens=2.00\nSurface=SAS\nRSolv=1.00')
        f.write('\n\n')

